/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigquery.storage.v1.it;

import com.google.api.gax.core.CredentialsProvider;
import com.google.api.gax.core.ExecutorProvider;
import com.google.api.gax.core.FixedCredentialsProvider;
import com.google.api.gax.core.InstantiatingExecutorProvider;
import com.google.api.gax.rpc.ServerStream;
import com.google.api.gax.rpc.UnauthenticatedException;
import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.RetryOption;
import com.google.cloud.ServiceOptions;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.DatasetInfo;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.LegacySQLTypeName;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.TimePartitioning;
import com.google.cloud.bigquery.storage.v1.BigQueryReadClient;
import com.google.cloud.bigquery.storage.v1.BigQueryReadSettings;
import com.google.cloud.bigquery.storage.v1.CreateReadSessionRequest;
import com.google.cloud.bigquery.storage.v1.DataFormat;
import com.google.cloud.bigquery.storage.v1.ReadRowsRequest;
import com.google.cloud.bigquery.storage.v1.ReadRowsResponse;
import com.google.cloud.bigquery.storage.v1.ReadSession;
import com.google.cloud.bigquery.storage.v1.ReadStream;
import com.google.cloud.bigquery.storage.v1.it.BigQueryResource;
import com.google.cloud.bigquery.storage.v1.it.SimpleRowReader;
import com.google.cloud.bigquery.testing.RemoteBigQueryHelper;
import com.google.common.base.Preconditions;
import com.google.common.truth.Truth;
import com.google.protobuf.Timestamp;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.logging.Logger;
import org.apache.avro.Conversions;
import org.apache.avro.LogicalTypes;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericData;
import org.apache.avro.generic.GenericRecordBuilder;
import org.apache.avro.util.Utf8;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.threeten.bp.Duration;
import org.threeten.bp.Instant;
import org.threeten.bp.LocalDate;
import org.threeten.bp.LocalTime;
import org.threeten.bp.ZoneId;
import org.threeten.bp.ZoneOffset;
import org.threeten.bp.ZonedDateTime;
import org.threeten.bp.format.DateTimeFormatter;

public class ITBigQueryStorageTest {
    private static final Logger LOG = Logger.getLogger(ITBigQueryStorageTest.class.getName());
    private static final String DATASET = RemoteBigQueryHelper.generateDatasetName();
    private static final String DESCRIPTION = "BigQuery Storage Java client test dataset";
    private static BigQueryReadClient client;
    private static String parentProjectId;
    private static BigQuery bigquery;
    private static final String FAKE_JSON_CRED_WITH_GOOGLE_DOMAIN = "{\n  \"private_key_id\": \"somekeyid\",\n  \"private_key\": \"-----BEGIN PRIVATE KEY-----\\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQC+K2hSuFpAdrJI\\nnCgcDz2M7t7bjdlsadsasad+fvRSW6TjNQZ3p5LLQY1kSZRqBqylRkzteMOyHgaR\\n0Pmxh3ILCND5men43j3h4eDbrhQBuxfEMalkG92sL+PNQSETY2tnvXryOvmBRwa/\\nQP/9dJfIkIDJ9Fw9N4Bhhhp6mCcRpdQjV38H7JsyJ7lih/oNjECgYAt\\nknddadwkwewcVxHFhcZJO+XWf6ofLUXpRwiTZakGMn8EE1uVa2LgczOjwWHGi99MFjxSer5m9\\n1tCa3/KEGKiS/YL71JvjwX3mb+cewlkcmweBKZHM2JPTk0ZednFSpVZMtycjkbLa\\ndYOS8V85AgMBewECggEBAKksaldajfDZDV6nGqbFjMiizAKJolr/M3OQw16K6o3/\\n0S31xIe3sSlgW0+UbYlF4U8KifhManD1apVSC3csafaspP4RZUHFhtBywLO9pR5c\\nr6S5aLp+gPWFyIp1pfXbWGvc5VY/v9x7ya1VEa6rXvLsKupSeWAW4tMj3eo/64ge\\nsdaceaLYw52KeBYiT6+vpsnYrEkAHO1fF/LavbLLOFJmFTMxmsNaG0tuiJHgjshB\\n82DpMCbXG9YcCgI/DbzuIjsdj2JC1cascSP//3PmefWysucBQe7Jryb6NQtASmnv\\nCdDw/0jmZTEjpe4S1lxfHplAhHFtdgYTvyYtaLZiVVkCgYEA8eVpof2rceecw/I6\\n5ng1q3Hl2usdWV/4mZMvR0fOemacLLfocX6IYxT1zA1FFJlbXSRsJMf/Qq39mOR2\\nSpW+hr4jCoHeRVYLgsbggtrevGmILAlNoqCMpGZ6vDmJpq6ECV9olliDvpPgWOP+\\nmYPDreFBGxWvQrADNbRt2dmGsrsCgYEAyUHqB2wvJHFqdmeBsaacewzV8x9WgmeX\\ngUIi9REwXlGDW0Mz50dxpxcKCAYn65+7TCnY5O/jmL0VRxU1J2mSWyWTo1C+17L0\\n3fUqjxL1pkefwecxwecvC+gFFYdJ4CQ/MHHXU81Lwl1iWdFCd2UoGddYaOF+KNeM\\nHC7cmqra+JsCgYEAlUNywzq8nUg7282E+uICfCB0LfwejuymR93CtsFgb7cRd6ak\\nECR8FGfCpH8ruWJINllbQfcHVCX47ndLZwqv3oVFKh6pAS/vVI4dpOepP8++7y1u\\ncoOvtreXCX6XqfrWDtKIvv0vjlHBhhhp6mCcRpdQjV38H7JsyJ7lih/oNjECgYAt\\nkndj5uNl5SiuVxHFhcZJO+XWf6ofLUregtevZakGMn8EE1uVa2AY7eafmoU/nZPT\\n00YB0TBATdCbn/nBSuKDESkhSg9s2GEKQZG5hBmL5uCMfo09z3SfxZIhJdlerreP\\nJ7gSidI12N+EZxYd4xIJh/HFDgp7RRO87f+WJkofMQKBgGTnClK1VMaCRbJZPriw\\nEfeFCoOX75MxKwXs6xgrw4W//AYGGUjDt83lD6AZP6tws7gJ2IwY/qP7+lyhjEqN\\nHtfPZRGFkGZsdaksdlaksd323423d+15/UvrlRSFPNj1tWQmNKkXyRDW4IG1Oa2p\\nrALStNBx5Y9t0/LQnFI4w3aG\\n-----END PRIVATE KEY-----\\n\",\n  \"project_id\": \"someprojectid\",\n  \"client_email\": \"someclientid@developer.gserviceaccount.com\",\n  \"client_id\": \"someclientid.apps.googleusercontent.com\",\n  \"type\": \"service_account\",\n  \"universe_domain\": \"googleapis.com\"\n}";
    private static final String FAKE_JSON_CRED_WITH_INVALID_DOMAIN = "{\n  \"private_key_id\": \"somekeyid\",\n  \"private_key\": \"-----BEGIN PRIVATE KEY-----\\nMIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQC+K2hSuFpAdrJI\\nnCgcDz2M7t7bjdlsadsasad+fvRSW6TjNQZ3p5LLQY1kSZRqBqylRkzteMOyHgaR\\n0Pmxh3ILCND5men43j3h4eDbrhQBuxfEMalkG92sL+PNQSETY2tnvXryOvmBRwa/\\nQP/9dJfIkIDJ9Fw9N4Bhhhp6mCcRpdQjV38H7JsyJ7lih/oNjECgYAt\\nknddadwkwewcVxHFhcZJO+XWf6ofLUXpRwiTZakGMn8EE1uVa2LgczOjwWHGi99MFjxSer5m9\\n1tCa3/KEGKiS/YL71JvjwX3mb+cewlkcmweBKZHM2JPTk0ZednFSpVZMtycjkbLa\\ndYOS8V85AgMBewECggEBAKksaldajfDZDV6nGqbFjMiizAKJolr/M3OQw16K6o3/\\n0S31xIe3sSlgW0+UbYlF4U8KifhManD1apVSC3csafaspP4RZUHFhtBywLO9pR5c\\nr6S5aLp+gPWFyIp1pfXbWGvc5VY/v9x7ya1VEa6rXvLsKupSeWAW4tMj3eo/64ge\\nsdaceaLYw52KeBYiT6+vpsnYrEkAHO1fF/LavbLLOFJmFTMxmsNaG0tuiJHgjshB\\n82DpMCbXG9YcCgI/DbzuIjsdj2JC1cascSP//3PmefWysucBQe7Jryb6NQtASmnv\\nCdDw/0jmZTEjpe4S1lxfHplAhHFtdgYTvyYtaLZiVVkCgYEA8eVpof2rceecw/I6\\n5ng1q3Hl2usdWV/4mZMvR0fOemacLLfocX6IYxT1zA1FFJlbXSRsJMf/Qq39mOR2\\nSpW+hr4jCoHeRVYLgsbggtrevGmILAlNoqCMpGZ6vDmJpq6ECV9olliDvpPgWOP+\\nmYPDreFBGxWvQrADNbRt2dmGsrsCgYEAyUHqB2wvJHFqdmeBsaacewzV8x9WgmeX\\ngUIi9REwXlGDW0Mz50dxpxcKCAYn65+7TCnY5O/jmL0VRxU1J2mSWyWTo1C+17L0\\n3fUqjxL1pkefwecxwecvC+gFFYdJ4CQ/MHHXU81Lwl1iWdFCd2UoGddYaOF+KNeM\\nHC7cmqra+JsCgYEAlUNywzq8nUg7282E+uICfCB0LfwejuymR93CtsFgb7cRd6ak\\nECR8FGfCpH8ruWJINllbQfcHVCX47ndLZwqv3oVFKh6pAS/vVI4dpOepP8++7y1u\\ncoOvtreXCX6XqfrWDtKIvv0vjlHBhhhp6mCcRpdQjV38H7JsyJ7lih/oNjECgYAt\\nkndj5uNl5SiuVxHFhcZJO+XWf6ofLUregtevZakGMn8EE1uVa2AY7eafmoU/nZPT\\n00YB0TBATdCbn/nBSuKDESkhSg9s2GEKQZG5hBmL5uCMfo09z3SfxZIhJdlerreP\\nJ7gSidI12N+EZxYd4xIJh/HFDgp7RRO87f+WJkofMQKBgGTnClK1VMaCRbJZPriw\\nEfeFCoOX75MxKwXs6xgrw4W//AYGGUjDt83lD6AZP6tws7gJ2IwY/qP7+lyhjEqN\\nHtfPZRGFkGZsdaksdlaksd323423d+15/UvrlRSFPNj1tWQmNKkXyRDW4IG1Oa2p\\nrALStNBx5Y9t0/LQnFI4w3aG\\n-----END PRIVATE KEY-----\\n\",\n  \"project_id\": \"someprojectid\",\n  \"client_email\": \"someclientid@developer.gserviceaccount.com\",\n  \"client_id\": \"someclientid.apps.googleusercontent.com\",\n  \"type\": \"service_account\",\n  \"universe_domain\": \"fake.domain\"\n}";

    @BeforeClass
    public static void beforeClass() throws IOException {
        client = BigQueryReadClient.create();
        parentProjectId = String.format("projects/%s", ServiceOptions.getDefaultProjectId());
        LOG.info(String.format("%s tests running with parent project: %s", ITBigQueryStorageTest.class.getSimpleName(), parentProjectId));
        RemoteBigQueryHelper bigqueryHelper = RemoteBigQueryHelper.create();
        bigquery = (BigQuery)bigqueryHelper.getOptions().getService();
        DatasetInfo datasetInfo = DatasetInfo.newBuilder((String)DATASET).setDescription(DESCRIPTION).build();
        bigquery.create(datasetInfo, new BigQuery.DatasetOption[0]);
        LOG.info("Created test dataset: " + DATASET);
    }

    @AfterClass
    public static void afterClass() {
        if (client != null) {
            client.close();
        }
        if (bigquery != null) {
            RemoteBigQueryHelper.forceDelete((BigQuery)bigquery, (String)DATASET);
            LOG.info("Deleted test dataset: " + DATASET);
        }
    }

    @Test
    public void testSimpleReadAvro() {
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        ReadSession session = client.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.AVRO).build(), 1);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table '%s' CreateReadSession response:%n%s", table, session.toString()), (long)1L, (long)session.getStreamsCount());
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(session.getStreams(0).getName()).build();
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)164656L, (long)rowCount);
    }

    @Test
    public void testSimpleReadArrow() {
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        ReadSession session = client.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.ARROW).build(), 1);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table '%s' CreateReadSession response:%n%s", table, session.toString()), (long)1L, (long)session.getStreamsCount());
        Preconditions.checkState((session.getStreamsCount() > 0 ? 1 : 0) != 0);
        String streamName = session.getStreams(0).getName();
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(streamName).build();
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            Preconditions.checkState((boolean)response.hasArrowRecordBatch());
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)164656L, (long)rowCount);
    }

    @Test
    public void testRangeType() throws InterruptedException {
        String tableName = "test_range_type";
        TableId tableId = TableId.of((String)DATASET, (String)tableName);
        QueryJobConfiguration createTable = QueryJobConfiguration.newBuilder((String)String.format("CREATE TABLE %s AS SELECT RANGE(DATE '2020-01-01', DATE '2020-12-31') as date, \nRANGE(DATETIME '2020-01-01T12:00:00', DATETIME '2020-12-31T12:00:00') as datetime, \nRANGE(TIMESTAMP '2014-01-01 07:00:00.000000+00:00', TIMESTAMP '2015-01-01 07:00:00.000000+00:00') as timestamp", tableName)).setDefaultDataset(DatasetId.of((String)DATASET)).setUseLegacySql(Boolean.valueOf(false)).build();
        bigquery.query(createTable, new BigQuery.JobOption[0]);
        String table = BigQueryResource.FormatTableResource(ServiceOptions.getDefaultProjectId(), DATASET, tableId.getTable());
        ReadSession session = client.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.ARROW).build(), 1);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table '%s' CreateReadSession response:%n%s", table, session.toString()), (long)1L, (long)session.getStreamsCount());
        Preconditions.checkState((session.getStreamsCount() > 0 ? 1 : 0) != 0);
        String streamName = session.getStreams(0).getName();
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(streamName).build();
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            Preconditions.checkState((boolean)response.hasArrowRecordBatch());
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)1L, (long)rowCount);
    }

    @Test
    public void testSimpleReadAndResume() {
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        ReadSession session = client.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.AVRO).build(), 1);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table '%s' CreateReadSession response:%n%s", table, session.toString()), (long)1L, (long)session.getStreamsCount());
        long rowCount = this.ReadStreamToOffset(session.getStreams(0), 34846L);
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(session.getStreams(0).getName()).setOffset(rowCount).build();
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)164656L, (long)rowCount);
    }

    @Test
    public void testFilter() throws IOException {
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        ReadSession.TableReadOptions options = ReadSession.TableReadOptions.newBuilder().setRowRestriction("word_count > 100").build();
        CreateReadSessionRequest request = CreateReadSessionRequest.newBuilder().setParent(parentProjectId).setMaxStreamCount(1).setReadSession(ReadSession.newBuilder().setTable(table).setReadOptions(options).setDataFormat(DataFormat.AVRO).build()).build();
        ReadSession session = client.createReadSession(request);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table '%s' CreateReadSession response:%n%s", table, session.toString()), (long)1L, (long)session.getStreamsCount());
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(session.getStreams(0).getName()).build();
        SimpleRowReader reader = new SimpleRowReader(new Schema.Parser().parse(session.getAvroSchema().getSchema()));
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
            reader.processRows(response.getAvroRows(), new SimpleRowReader.AvroRowConsumer(){

                @Override
                public void accept(GenericData.Record record) {
                    Long wordCount = (Long)record.get("word_count");
                    Truth.assertWithMessage((String)"Row not matching expectations: %s", (Object[])new Object[]{record.toString()}).that(wordCount).isGreaterThan((Comparable)Long.valueOf(100L));
                }
            });
        }
        Assert.assertEquals((long)1333L, (long)rowCount);
    }

    @Test
    public void testColumnSelection() throws IOException {
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        ReadSession.TableReadOptions options = ReadSession.TableReadOptions.newBuilder().addSelectedFields("word").addSelectedFields("word_count").setRowRestriction("word_count > 100").build();
        CreateReadSessionRequest request = CreateReadSessionRequest.newBuilder().setParent(parentProjectId).setMaxStreamCount(1).setReadSession(ReadSession.newBuilder().setTable(table).setReadOptions(options).setDataFormat(DataFormat.AVRO).build()).build();
        ReadSession session = client.createReadSession(request);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table '%s' CreateReadSession response:%n%s", table, session.toString()), (long)1L, (long)session.getStreamsCount());
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(session.getStreams(0).getName()).build();
        org.apache.avro.Schema avroSchema = new Schema.Parser().parse(session.getAvroSchema().getSchema());
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)2L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)avroSchema.getField("word").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("word_count").schema().getType());
        SimpleRowReader reader = new SimpleRowReader(avroSchema);
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
            reader.processRows(response.getAvroRows(), new SimpleRowReader.AvroRowConsumer(){

                @Override
                public void accept(GenericData.Record record) {
                    String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
                    Long wordCount = (Long)record.get("word_count");
                    Truth.assertWithMessage((String)rowAssertMessage).that(wordCount).isGreaterThan((Comparable)Long.valueOf(100L));
                    Utf8 word = (Utf8)record.get("word");
                    Truth.assertWithMessage((String)rowAssertMessage).that(Integer.valueOf(word.length())).isGreaterThan((Comparable)Integer.valueOf(0));
                }
            });
        }
        Assert.assertEquals((long)1333L, (long)rowCount);
    }

    @Test
    public void testReadAtSnapshot() throws InterruptedException, IOException {
        Field intFieldSchema = Field.newBuilder((String)"col", (LegacySQLTypeName)LegacySQLTypeName.INTEGER, (Field[])new Field[0]).setMode(Field.Mode.REQUIRED).setDescription("IntegerDescription").build();
        Schema tableSchema = Schema.of((Field[])new Field[]{intFieldSchema});
        TableId testTableId = TableId.of((String)DATASET, (String)"test_read_snapshot");
        bigquery.create(TableInfo.of((TableId)testTableId, (TableDefinition)StandardTableDefinition.of((Schema)tableSchema)), new BigQuery.TableOption[0]);
        testTableId.toString();
        Job firstJob = this.RunQueryAppendJobAndExpectSuccess(testTableId, "SELECT 1 AS col");
        Job secondJob = this.RunQueryAppendJobAndExpectSuccess(testTableId, "SELECT 2 AS col");
        String table = BigQueryResource.FormatTableResource(ServiceOptions.getDefaultProjectId(), DATASET, testTableId.getTable());
        final ArrayList rowsAfterFirstSnapshot = new ArrayList();
        this.ProcessRowsAtSnapshot(table, firstJob.getStatistics().getEndTime(), null, new SimpleRowReader.AvroRowConsumer(){

            @Override
            public void accept(GenericData.Record record) {
                rowsAfterFirstSnapshot.add((Long)record.get("col"));
            }
        });
        Assert.assertEquals(Arrays.asList(1L), rowsAfterFirstSnapshot);
        final ArrayList rowsAfterSecondSnapshot = new ArrayList();
        this.ProcessRowsAtSnapshot(table, secondJob.getStatistics().getEndTime(), null, new SimpleRowReader.AvroRowConsumer(){

            @Override
            public void accept(GenericData.Record record) {
                rowsAfterSecondSnapshot.add((Long)record.get("col"));
            }
        });
        Collections.sort(rowsAfterSecondSnapshot);
        Assert.assertEquals(Arrays.asList(1L, 2L), rowsAfterSecondSnapshot);
    }

    @Test
    public void testColumnPartitionedTableByDateField() throws InterruptedException, IOException {
        String partitionedTableName = "test_column_partition_table_by_date";
        String createTableStatement = String.format(" CREATE TABLE %s.%s (num_field INT64, date_field DATE)  PARTITION BY date_field  OPTIONS(    description=\"a table partitioned by date_field\"  ) AS    SELECT 1, CAST(\"2019-01-01\" AS DATE)   UNION ALL   SELECT 2, CAST(\"2019-01-02\" AS DATE)   UNION ALL   SELECT 3, CAST(\"2019-01-03\" AS DATE)", DATASET, partitionedTableName);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        String table = BigQueryResource.FormatTableResource(ServiceOptions.getDefaultProjectId(), DATASET, partitionedTableName);
        List<GenericData.Record> unfilteredRows = this.ReadAllRows(table, null);
        Assert.assertEquals((String)("Actual rows read: " + unfilteredRows.toString()), (long)3L, (long)unfilteredRows.size());
        List<GenericData.Record> partitionFilteredRows = this.ReadAllRows(table, "date_field = CAST(\"2019-01-02\" AS DATE)");
        Assert.assertEquals((String)("Actual rows read: " + partitionFilteredRows.toString()), (long)1L, (long)partitionFilteredRows.size());
        Assert.assertEquals((Object)2L, (Object)partitionFilteredRows.get(0).get("num_field"));
    }

    @Test
    public void testIngestionTimePartitionedTable() throws InterruptedException, IOException {
        Field intFieldSchema = Field.newBuilder((String)"num_field", (LegacySQLTypeName)LegacySQLTypeName.INTEGER, (Field[])new Field[0]).setMode(Field.Mode.REQUIRED).setDescription("IntegerDescription").build();
        Schema tableSchema = Schema.of((Field[])new Field[]{intFieldSchema});
        TableId testTableId = TableId.of((String)DATASET, (String)"test_date_partitioned_table");
        bigquery.create(TableInfo.of((TableId)testTableId, (TableDefinition)StandardTableDefinition.newBuilder().setTimePartitioning(TimePartitioning.of((TimePartitioning.Type)TimePartitioning.Type.DAY)).setSchema(tableSchema).build()), new BigQuery.TableOption[0]);
        this.RunQueryAppendJobAndExpectSuccess(TableId.of((String)DATASET, (String)(testTableId.getTable() + "$20190101")), "SELECT 1 AS num_field");
        this.RunQueryAppendJobAndExpectSuccess(TableId.of((String)DATASET, (String)(testTableId.getTable() + "$20190102")), "SELECT 2 AS num_field");
        String table = BigQueryResource.FormatTableResource(ServiceOptions.getDefaultProjectId(), testTableId.getDataset(), testTableId.getTable());
        List<GenericData.Record> unfilteredRows = this.ReadAllRows(table, null);
        Assert.assertEquals((String)("Actual rows read: " + unfilteredRows.toString()), (long)2L, (long)unfilteredRows.size());
        List<GenericData.Record> partitionFilteredRows = this.ReadAllRows(table, "_PARTITIONDATE > \"2019-01-01\"");
        Assert.assertEquals((String)("Actual rows read: " + partitionFilteredRows.toString()), (long)1L, (long)partitionFilteredRows.size());
        Assert.assertEquals((Object)2L, (Object)partitionFilteredRows.get(0).get("num_field"));
    }

    @Test
    public void testBasicSqlTypes() throws InterruptedException, IOException {
        String tableName = "test_basic_sql_types";
        String createTableStatement = String.format(" CREATE TABLE %s.%s  (int_field INT64 NOT NULL,  num_field NUMERIC NOT NULL,  float_field FLOAT64 NOT NULL,  bool_field BOOL NOT NULL,  str_field STRING NOT NULL,  bytes_field BYTES NOT NULL)  OPTIONS(    description=\"a table with basic column types\"  ) AS    SELECT      17,     CAST(1234.56 AS NUMERIC),     6.547678,     TRUE,     \"String field value\",     b\"\u0430\u0431\u0432\u0433\u0434\"", DATASET, tableName);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        String table = BigQueryResource.FormatTableResource(ServiceOptions.getDefaultProjectId(), DATASET, tableName);
        List<GenericData.Record> rows = this.ReadAllRows(table, null);
        Assert.assertEquals((String)("Actual rows read: " + rows.toString()), (long)1L, (long)rows.size());
        GenericData.Record record = rows.get(0);
        org.apache.avro.Schema avroSchema = record.getSchema();
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)6L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("int_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (long)17L, (long)((Long)record.get("int_field")));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.BYTES, (Object)avroSchema.getField("num_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)LogicalTypes.decimal((int)38, (int)9), (Object)avroSchema.getField("num_field").schema().getLogicalType());
        BigDecimal actual_num_field = new Conversions.DecimalConversion().fromBytes((ByteBuffer)record.get("num_field"), avroSchema, avroSchema.getField("num_field").schema().getLogicalType());
        Assert.assertEquals((String)rowAssertMessage, (Object)BigDecimal.valueOf(1234560000000L, 9), (Object)actual_num_field);
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.DOUBLE, (Object)avroSchema.getField("float_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (double)6.547678, (double)((Double)record.get("float_field")), (double)1.0E-4);
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.BOOLEAN, (Object)avroSchema.getField("bool_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (Object)true, (Object)record.get("bool_field"));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)avroSchema.getField("str_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (Object)new Utf8("String field value"), (Object)record.get("str_field"));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.BYTES, (Object)avroSchema.getField("bytes_field").schema().getType());
        Assert.assertArrayEquals((String)rowAssertMessage, (byte[])Utf8.getBytesFor((String)"\u0430\u0431\u0432\u0433\u0434"), (byte[])((ByteBuffer)record.get("bytes_field")).array());
    }

    @Test
    public void testDateAndTimeSqlTypes() throws InterruptedException, IOException {
        String tableName = "test_date_and_time_sql_types";
        String createTableStatement = String.format(" CREATE TABLE %s.%s  (date_field DATE NOT NULL,  datetime_field DATETIME NOT NULL,  time_field TIME NOT NULL,  timestamp_field TIMESTAMP NOT NULL) OPTIONS(    description=\"a table with date and time column types\"  ) AS    SELECT      CAST(\"2019-05-31\" AS DATE),     CAST(\"2019-04-30 21:47:59.999999\" AS DATETIME),     CAST(\"21:47:59.999999\" AS TIME),     CAST(\"2019-04-30 19:24:19.123456 UTC\" AS TIMESTAMP)", DATASET, tableName);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        String table = BigQueryResource.FormatTableResource(ServiceOptions.getDefaultProjectId(), DATASET, tableName);
        List<GenericData.Record> rows = this.ReadAllRows(table, null);
        Assert.assertEquals((String)("Actual rows read: " + rows.toString()), (long)1L, (long)rows.size());
        GenericData.Record record = rows.get(0);
        org.apache.avro.Schema avroSchema = record.getSchema();
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)4L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.INT, (Object)avroSchema.getField("date_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)LogicalTypes.date(), (Object)avroSchema.getField("date_field").schema().getLogicalType());
        Assert.assertEquals((String)rowAssertMessage, (Object)LocalDate.of((int)2019, (int)5, (int)31), (Object)LocalDate.ofEpochDay((long)((Integer)record.get("date_field")).intValue()));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)avroSchema.getField("datetime_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"datetime", (Object)avroSchema.getField("datetime_field").schema().getObjectProp("logicalType"));
        Assert.assertEquals((String)rowAssertMessage, (Object)new Utf8("2019-04-30T21:47:59.999999"), (Object)((Utf8)record.get("datetime_field")));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("time_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)LogicalTypes.timeMicros(), (Object)avroSchema.getField("time_field").schema().getLogicalType());
        Assert.assertEquals((String)rowAssertMessage, (Object)LocalTime.of((int)21, (int)47, (int)59, (int)999999000), (Object)LocalTime.ofNanoOfDay((long)(1000L * (Long)record.get("time_field"))));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("timestamp_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)LogicalTypes.timestampMicros(), (Object)avroSchema.getField("timestamp_field").schema().getLogicalType());
        ZonedDateTime expected_timestamp = ZonedDateTime.parse((CharSequence)"2019-04-30T19:24:19Z", (DateTimeFormatter)DateTimeFormatter.ISO_INSTANT.withZone((ZoneId)ZoneOffset.UTC)).withNano(123456000);
        long actual_timestamp_micros = (Long)record.get("timestamp_field");
        ZonedDateTime actual_timestamp = ZonedDateTime.ofInstant((Instant)Instant.ofEpochSecond((long)(actual_timestamp_micros / 1000000L), (long)(actual_timestamp_micros % 1000000L * 1000L)), (ZoneId)ZoneOffset.UTC);
        Assert.assertEquals((String)rowAssertMessage, (Object)expected_timestamp, (Object)actual_timestamp);
    }

    @Test
    public void testGeographySqlType() throws InterruptedException, IOException {
        String tableName = "test_geography_sql_type";
        String createTableStatement = String.format(" CREATE TABLE %s.%s  (geo_field GEOGRAPHY NOT NULL) OPTIONS(    description=\"a table with a geography column type\"  ) AS    SELECT ST_GEOGPOINT(1.1, 2.2)", DATASET, tableName);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        String table = BigQueryResource.FormatTableResource(ServiceOptions.getDefaultProjectId(), DATASET, tableName);
        List<GenericData.Record> rows = this.ReadAllRows(table, null);
        Assert.assertEquals((String)("Actual rows read: " + rows.toString()), (long)1L, (long)rows.size());
        GenericData.Record record = rows.get(0);
        org.apache.avro.Schema avroSchema = record.getSchema();
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)1L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)avroSchema.getField("geo_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"GEOGRAPHY", (Object)avroSchema.getField("geo_field").schema().getObjectProp("sqlType"));
        Assert.assertEquals((String)rowAssertMessage, (Object)new Utf8("POINT(1.1 2.2)"), (Object)((Utf8)record.get("geo_field")));
    }

    @Test
    public void testStructAndArraySqlTypes() throws InterruptedException, IOException {
        String tableName = "test_struct_and_array_sql_types";
        String createTableStatement = String.format(" CREATE TABLE %s.%s  (array_field ARRAY<INT64>,  struct_field STRUCT<int_field INT64 NOT NULL, str_field STRING NOT NULL> NOT NULL) OPTIONS(    description=\"a table with array and time column types\"  ) AS    SELECT      [1, 2, 3],     (10, 'abc')", DATASET, tableName);
        this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)createTableStatement).build());
        String table = BigQueryResource.FormatTableResource(ServiceOptions.getDefaultProjectId(), DATASET, tableName);
        List<GenericData.Record> rows = this.ReadAllRows(table, null);
        Assert.assertEquals((String)("Actual rows read: " + rows.toString()), (long)1L, (long)rows.size());
        GenericData.Record record = rows.get(0);
        org.apache.avro.Schema avroSchema = record.getSchema();
        String actualSchemaMessage = String.format("Unexpected schema. Actual schema:%n%s", avroSchema.toString(true));
        String rowAssertMessage = String.format("Row not matching expectations: %s", record.toString());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)avroSchema.getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)"__root__", (Object)avroSchema.getName());
        Assert.assertEquals((String)actualSchemaMessage, (long)2L, (long)avroSchema.getFields().size());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.ARRAY, (Object)avroSchema.getField("array_field").schema().getType());
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)avroSchema.getField("array_field").schema().getElementType().getType());
        Assert.assertArrayEquals((String)rowAssertMessage, (Object[])new Long[]{1L, 2L, 3L}, (Object[])((GenericData.Array)record.get("array_field")).toArray((Object[])new Long[0]));
        org.apache.avro.Schema structSchema = avroSchema.getField("struct_field").schema();
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.RECORD, (Object)structSchema.getType());
        GenericData.Record structRecord = (GenericData.Record)record.get("struct_field");
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.LONG, (Object)structSchema.getField("int_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (long)10L, (long)((Long)structRecord.get("int_field")));
        Assert.assertEquals((String)actualSchemaMessage, (Object)Schema.Type.STRING, (Object)structSchema.getField("str_field").schema().getType());
        Assert.assertEquals((String)rowAssertMessage, (Object)new Utf8("abc"), (Object)structRecord.get("str_field"));
    }

    @Test
    public void testSimpleReadWithBackgroundExecutorProvider() throws IOException {
        BigQueryReadSettings bigQueryReadSettings = ((BigQueryReadSettings.Builder)BigQueryReadSettings.newBuilder().setBackgroundExecutorProvider((ExecutorProvider)InstantiatingExecutorProvider.newBuilder().setExecutorThreadCount(14).build())).build();
        client = BigQueryReadClient.create((BigQueryReadSettings)bigQueryReadSettings);
        Assert.assertTrue((boolean)(client.getStub().getStubSettings().getBackgroundExecutorProvider() instanceof InstantiatingExecutorProvider));
        Assert.assertEquals((long)14L, (long)((InstantiatingExecutorProvider)client.getStub().getStubSettings().getBackgroundExecutorProvider()).getExecutorThreadCount());
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        ReadSession session = client.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.AVRO).build(), 1);
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table '%s' CreateReadSession response:%n%s", table, session.toString()), (long)1L, (long)session.getStreamsCount());
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(session.getStreams(0).getName()).build();
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)164656L, (long)rowCount);
    }

    @Test
    public void testUniverseDomainWithInvalidUniverseDomain() throws IOException {
        BigQueryReadSettings bigQueryReadSettings = ((BigQueryReadSettings.Builder)((BigQueryReadSettings.Builder)BigQueryReadSettings.newBuilder().setCredentialsProvider((CredentialsProvider)FixedCredentialsProvider.create((Credentials)ITBigQueryStorageTest.loadCredentials(FAKE_JSON_CRED_WITH_GOOGLE_DOMAIN)))).setUniverseDomain("invalid.domain")).build();
        BigQueryReadClient localClient = BigQueryReadClient.create((BigQueryReadSettings)bigQueryReadSettings);
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        try {
            localClient.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.AVRO).build(), 1);
            Assert.fail((String)"RPCs to invalid universe domain should fail");
        }
        catch (UnauthenticatedException e) {
            Truth.assertThat((Boolean)e.getMessage().contains("does not match the universe domain found in the credentials")).isTrue();
        }
        localClient.close();
    }

    @Test
    public void testInvalidUniverseDomainWithMismatchCredentials() throws IOException {
        BigQueryReadSettings bigQueryReadSettings = ((BigQueryReadSettings.Builder)((BigQueryReadSettings.Builder)BigQueryReadSettings.newBuilder().setCredentialsProvider((CredentialsProvider)FixedCredentialsProvider.create((Credentials)ITBigQueryStorageTest.loadCredentials(FAKE_JSON_CRED_WITH_INVALID_DOMAIN)))).setUniverseDomain("invalid.domain")).build();
        BigQueryReadClient localClient = BigQueryReadClient.create((BigQueryReadSettings)bigQueryReadSettings);
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        try {
            ReadSession session = localClient.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.AVRO).build(), 1);
            Assert.fail((String)"RPCs to invalid universe domain should fail");
        }
        catch (UnauthenticatedException e) {
            Truth.assertThat((Boolean)e.getMessage().contains("does not match the universe domain found in the credentials")).isTrue();
        }
        localClient.close();
    }

    @Test
    public void testUniverseDomainWithMatchingDomain() throws IOException {
        BigQueryReadSettings bigQueryReadSettings = ((BigQueryReadSettings.Builder)BigQueryReadSettings.newBuilder().setUniverseDomain("googleapis.com")).build();
        BigQueryReadClient localClient = BigQueryReadClient.create((BigQueryReadSettings)bigQueryReadSettings);
        String table = BigQueryResource.FormatTableResource("bigquery-public-data", "samples", "shakespeare");
        ReadSession session = localClient.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.AVRO).build(), 1);
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(session.getStreams(0).getName()).build();
        long rowCount = 0L;
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)164656L, (long)rowCount);
        localClient.close();
    }

    public void testUniverseDomain() throws IOException {
        BigQueryReadSettings bigQueryReadSettings = ((BigQueryReadSettings.Builder)BigQueryReadSettings.newBuilder().setUniverseDomain("apis-tpclp.goog")).build();
        BigQueryReadClient localClient = BigQueryReadClient.create((BigQueryReadSettings)bigQueryReadSettings);
        String table = BigQueryResource.FormatTableResource("google-tpc-testing-environment:cloudsdk-test-project", "tpc_demo_dataset", "new_table");
        ReadSession session = localClient.createReadSession(parentProjectId, ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.AVRO).build(), 1);
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(session.getStreams(0).getName()).build();
        long rowCount = 0L;
        ServerStream stream = localClient.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            rowCount += response.getRowCount();
        }
        Assert.assertEquals((long)1L, (long)rowCount);
        localClient.close();
    }

    private long ReadStreamToOffset(ReadStream readStream, long rowOffset) {
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(readStream.getName()).build();
        long rowCount = 0L;
        ServerStream serverStream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : serverStream) {
            if ((rowCount += response.getRowCount()) < rowOffset) continue;
            return rowOffset;
        }
        return rowCount;
    }

    private void ProcessRowsAtSnapshot(String table, Long snapshotInMillis, String filter, SimpleRowReader.AvroRowConsumer consumer) throws IOException {
        Preconditions.checkNotNull((Object)table);
        Preconditions.checkNotNull((Object)consumer);
        CreateReadSessionRequest.Builder createSessionRequestBuilder = CreateReadSessionRequest.newBuilder().setParent(parentProjectId).setMaxStreamCount(1).setReadSession(ReadSession.newBuilder().setTable(table).setDataFormat(DataFormat.AVRO).build());
        if (snapshotInMillis != null) {
            Timestamp snapshotTimestamp = Timestamp.newBuilder().setSeconds(snapshotInMillis / 1000L).setNanos((int)(snapshotInMillis % 1000L * 1000000L)).build();
            createSessionRequestBuilder.getReadSessionBuilder().setTableModifiers(ReadSession.TableModifiers.newBuilder().setSnapshotTime(snapshotTimestamp).build());
        }
        if (filter != null && !filter.isEmpty()) {
            createSessionRequestBuilder.getReadSessionBuilder().setReadOptions(ReadSession.TableReadOptions.newBuilder().setRowRestriction(filter).build());
        }
        ReadSession session = client.createReadSession(createSessionRequestBuilder.build());
        Assert.assertEquals((String)String.format("Did not receive expected number of streams for table '%s' CreateReadSession response:%n%s", table, session.toString()), (long)1L, (long)session.getStreamsCount());
        ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder().setReadStream(session.getStreams(0).getName()).build();
        SimpleRowReader reader = new SimpleRowReader(new Schema.Parser().parse(session.getAvroSchema().getSchema()));
        ServerStream stream = client.readRowsCallable().call((Object)readRowsRequest);
        for (ReadRowsResponse response : stream) {
            reader.processRows(response.getAvroRows(), consumer);
        }
    }

    List<GenericData.Record> ReadAllRows(String table, String filter) throws IOException {
        final ArrayList<GenericData.Record> rows = new ArrayList<GenericData.Record>();
        this.ProcessRowsAtSnapshot(table, null, filter, new SimpleRowReader.AvroRowConsumer(){

            @Override
            public void accept(GenericData.Record record) {
                rows.add(new GenericRecordBuilder(record).build());
            }
        });
        return rows;
    }

    private Job RunQueryAppendJobAndExpectSuccess(TableId destinationTableId, String query) throws InterruptedException {
        return this.RunQueryJobAndExpectSuccess(QueryJobConfiguration.newBuilder((String)query).setDestinationTable(destinationTableId).setUseQueryCache(Boolean.valueOf(false)).setUseLegacySql(Boolean.valueOf(false)).setWriteDisposition(JobInfo.WriteDisposition.WRITE_APPEND).build());
    }

    private Job RunQueryJobAndExpectSuccess(QueryJobConfiguration configuration) throws InterruptedException {
        Job job = bigquery.create(JobInfo.of((JobConfiguration)configuration), new BigQuery.JobOption[0]);
        Job completedJob = job.waitFor(new RetryOption[]{RetryOption.initialRetryDelay((Duration)Duration.ofSeconds((long)1L)), RetryOption.totalTimeout((Duration)Duration.ofMinutes((long)1L))});
        Assert.assertNotNull((Object)completedJob);
        Assert.assertNull((String)("Received a job status that is not a success: " + completedJob.getStatus().toString()), (Object)completedJob.getStatus().getError());
        return completedJob;
    }

    static GoogleCredentials loadCredentials(String credentialFile) {
        try {
            ByteArrayInputStream keyStream = new ByteArrayInputStream(credentialFile.getBytes());
            return GoogleCredentials.fromStream((InputStream)keyStream);
        }
        catch (IOException e) {
            Assert.fail((String)"Couldn't create fake JSON credentials.");
            return null;
        }
    }
}

