/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigtable.data.v2.stub.metrics;

import com.google.api.MonitoredResource;
import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutureCallback;
import com.google.api.core.ApiFutures;
import com.google.api.core.InternalApi;
import com.google.api.gax.core.CredentialsProvider;
import com.google.api.gax.core.FixedCredentialsProvider;
import com.google.api.gax.core.NoCredentialsProvider;
import com.google.auth.Credentials;
import com.google.cloud.bigtable.data.v2.stub.metrics.BigtableExporterUtils;
import com.google.cloud.monitoring.v3.MetricServiceClient;
import com.google.cloud.monitoring.v3.MetricServiceSettings;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.monitoring.v3.CreateTimeSeriesRequest;
import com.google.monitoring.v3.ProjectName;
import com.google.monitoring.v3.TimeSeries;
import com.google.protobuf.Empty;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.metrics.InstrumentType;
import io.opentelemetry.sdk.metrics.data.AggregationTemporality;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.export.MetricExporter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.threeten.bp.Duration;

@InternalApi
public final class BigtableCloudMonitoringExporter
implements MetricExporter {
    private static final Logger logger = Logger.getLogger(BigtableCloudMonitoringExporter.class.getName());
    private static final String MONITORING_ENDPOINT = (String)MoreObjects.firstNonNull((Object)System.getProperty("bigtable.test-monitoring-endpoint"), (Object)MetricServiceSettings.getDefaultEndpoint());
    private static final String APPLICATION_RESOURCE_PROJECT_ID = "project_id";
    private final MetricServiceClient client;
    private final String bigtableProjectId;
    private final String taskId;
    private final MonitoredResource applicationResource;
    private final AtomicBoolean isShutdown = new AtomicBoolean(false);
    private CompletableResultCode lastExportCode;
    private static final ImmutableList<String> BIGTABLE_TABLE_METRICS = (ImmutableList)ImmutableSet.of((Object)"operation_latencies", (Object)"attempt_latencies", (Object)"server_latencies", (Object)"first_response_latencies", (Object)"throttling_latencies", (Object)"application_latencies", (Object[])new String[]{"retry_count", "connectivity_error_count"}).stream().map(m -> "bigtable.googleapis.com/internal/client/" + m).collect(ImmutableList.toImmutableList());
    private static final ImmutableList<String> APPLICATION_METRICS = (ImmutableList)ImmutableSet.of((Object)"per_connection_error_count").stream().map(m -> "bigtable.googleapis.com/internal/client/" + m).collect(ImmutableList.toImmutableList());

    public static BigtableCloudMonitoringExporter create(String projectId, @Nullable Credentials credentials) throws IOException {
        MetricServiceSettings.Builder settingsBuilder = MetricServiceSettings.newBuilder();
        CredentialsProvider credentialsProvider = Optional.ofNullable(credentials).map(FixedCredentialsProvider::create).orElse((CredentialsProvider)NoCredentialsProvider.create());
        settingsBuilder.setCredentialsProvider(credentialsProvider);
        settingsBuilder.setEndpoint(MONITORING_ENDPOINT);
        Duration timeout = Duration.ofMinutes((long)1L);
        settingsBuilder.createServiceTimeSeriesSettings().setSimpleTimeoutNoRetries(timeout);
        MonitoredResource applicationResource = null;
        try {
            applicationResource = BigtableExporterUtils.detectResource();
        }
        catch (Exception e) {
            logger.log(Level.WARNING, "Failed to detect resource, will skip exporting application level metrics ", e);
        }
        return new BigtableCloudMonitoringExporter(projectId, MetricServiceClient.create((MetricServiceSettings)settingsBuilder.build()), applicationResource, BigtableExporterUtils.getDefaultTaskValue());
    }

    @VisibleForTesting
    BigtableCloudMonitoringExporter(String projectId, MetricServiceClient client, @Nullable MonitoredResource applicationResource, String taskId) {
        this.client = client;
        this.taskId = taskId;
        this.applicationResource = applicationResource;
        this.bigtableProjectId = projectId;
    }

    public CompletableResultCode export(Collection<MetricData> collection) {
        if (this.isShutdown.get()) {
            logger.log(Level.WARNING, "Exporter is shutting down");
            return CompletableResultCode.ofFailure();
        }
        CompletableResultCode bigtableExportCode = this.exportBigtableResourceMetrics(collection);
        CompletableResultCode applicationExportCode = this.exportApplicationResourceMetrics(collection);
        this.lastExportCode = CompletableResultCode.ofAll((Collection)ImmutableList.of((Object)applicationExportCode, (Object)bigtableExportCode));
        return this.lastExportCode;
    }

    private CompletableResultCode exportBigtableResourceMetrics(Collection<MetricData> collection) {
        List<TimeSeries> bigtableTimeSeries;
        List<MetricData> bigtableMetricData = collection.stream().filter(md -> BIGTABLE_TABLE_METRICS.contains((Object)md.getName())).collect(Collectors.toList());
        if (bigtableMetricData.isEmpty()) {
            return CompletableResultCode.ofSuccess();
        }
        if (!bigtableMetricData.stream().flatMap(metricData -> metricData.getData().getPoints().stream()).allMatch(pd -> this.bigtableProjectId.equals(BigtableExporterUtils.getProjectId(pd)))) {
            logger.log(Level.WARNING, "Metric data has different a projectId. Skip exporting.");
            return CompletableResultCode.ofFailure();
        }
        try {
            bigtableTimeSeries = BigtableExporterUtils.convertToBigtableTimeSeries(bigtableMetricData, this.taskId);
        }
        catch (Throwable e) {
            logger.log(Level.WARNING, "Failed to convert bigtable table metric data to cloud monitoring timeseries.", e);
            return CompletableResultCode.ofFailure();
        }
        ProjectName projectName = ProjectName.of((String)this.bigtableProjectId);
        CreateTimeSeriesRequest bigtableRequest = CreateTimeSeriesRequest.newBuilder().setName(projectName.toString()).addAllTimeSeries(bigtableTimeSeries).build();
        ApiFuture future = this.client.createServiceTimeSeriesCallable().futureCall((Object)bigtableRequest);
        final CompletableResultCode bigtableExportCode = new CompletableResultCode();
        ApiFutures.addCallback((ApiFuture)future, (ApiFutureCallback)new ApiFutureCallback<Empty>(){

            public void onFailure(Throwable throwable) {
                logger.log(Level.WARNING, "createServiceTimeSeries request failed for bigtable metrics. ", throwable);
                bigtableExportCode.fail();
            }

            public void onSuccess(Empty empty) {
                bigtableExportCode.succeed();
            }
        }, (Executor)MoreExecutors.directExecutor());
        return bigtableExportCode;
    }

    private CompletableResultCode exportApplicationResourceMetrics(Collection<MetricData> collection) {
        List<TimeSeries> timeSeries;
        if (this.applicationResource == null) {
            return CompletableResultCode.ofSuccess();
        }
        List<MetricData> metricData = collection.stream().filter(md -> APPLICATION_METRICS.contains((Object)md.getName())).collect(Collectors.toList());
        if (metricData.isEmpty()) {
            return CompletableResultCode.ofSuccess();
        }
        try {
            timeSeries = BigtableExporterUtils.convertToApplicationResourceTimeSeries(metricData, this.taskId, this.applicationResource);
        }
        catch (Throwable e) {
            logger.log(Level.WARNING, "Failed to convert application metric data to cloud monitoring timeseries.", e);
            return CompletableResultCode.ofFailure();
        }
        final CompletableResultCode exportCode = new CompletableResultCode();
        try {
            ProjectName projectName = ProjectName.of((String)this.applicationResource.getLabelsOrThrow(APPLICATION_RESOURCE_PROJECT_ID));
            CreateTimeSeriesRequest request = CreateTimeSeriesRequest.newBuilder().setName(projectName.toString()).addAllTimeSeries(timeSeries).build();
            ApiFuture gceOrGkeFuture = this.client.createServiceTimeSeriesCallable().futureCall((Object)request);
            ApiFutures.addCallback((ApiFuture)gceOrGkeFuture, (ApiFutureCallback)new ApiFutureCallback<Empty>(){

                public void onFailure(Throwable throwable) {
                    logger.log(Level.WARNING, "createServiceTimeSeries request failed for per connection error metrics.", throwable);
                    exportCode.fail();
                }

                public void onSuccess(Empty empty) {
                    exportCode.succeed();
                }
            }, (Executor)MoreExecutors.directExecutor());
        }
        catch (Exception e) {
            logger.log(Level.WARNING, "Failed to get projectName for application resource " + this.applicationResource);
            return CompletableResultCode.ofFailure();
        }
        return exportCode;
    }

    public CompletableResultCode flush() {
        if (this.lastExportCode != null) {
            return this.lastExportCode;
        }
        return CompletableResultCode.ofSuccess();
    }

    public CompletableResultCode shutdown() {
        if (!this.isShutdown.compareAndSet(false, true)) {
            logger.log(Level.WARNING, "shutdown is called multiple times");
            return CompletableResultCode.ofSuccess();
        }
        CompletableResultCode flushResult = this.flush();
        CompletableResultCode shutdownResult = new CompletableResultCode();
        flushResult.whenComplete(() -> {
            Throwable throwable = null;
            try {
                this.client.shutdown();
            }
            catch (Throwable e) {
                logger.log(Level.WARNING, "failed to shutdown the monitoring client", e);
                throwable = e;
            }
            if (throwable != null) {
                shutdownResult.fail();
            } else {
                shutdownResult.succeed();
            }
        });
        return CompletableResultCode.ofAll(Arrays.asList(flushResult, shutdownResult));
    }

    public AggregationTemporality getAggregationTemporality(InstrumentType instrumentType) {
        return AggregationTemporality.CUMULATIVE;
    }
}

