/*
 * Copyright 2021 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.cloud.compute.v1;

import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutures;
import com.google.api.core.BetaApi;
import com.google.api.gax.core.BackgroundResource;
import com.google.api.gax.longrunning.OperationFuture;
import com.google.api.gax.paging.AbstractFixedSizeCollection;
import com.google.api.gax.paging.AbstractPage;
import com.google.api.gax.paging.AbstractPagedListResponse;
import com.google.api.gax.rpc.OperationCallable;
import com.google.api.gax.rpc.PageContext;
import com.google.api.gax.rpc.UnaryCallable;
import com.google.cloud.compute.v1.stub.AutoscalersStub;
import com.google.cloud.compute.v1.stub.AutoscalersStubSettings;
import com.google.common.util.concurrent.MoreExecutors;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import javax.annotation.Generated;

// AUTO-GENERATED DOCUMENTATION AND CLASS.
/**
 * Service Description: The Autoscalers API.
 *
 * <p>This class provides the ability to make remote calls to the backing service through method
 * calls that map to API methods. Sample code to get started:
 *
 * <pre>{@code
 * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
 *   String project = "project-309310695";
 *   String zone = "zone3744684";
 *   String autoscaler = "autoscaler517258967";
 *   Autoscaler response = autoscalersClient.get(project, zone, autoscaler);
 * }
 * }</pre>
 *
 * <p>Note: close() needs to be called on the AutoscalersClient object to clean up resources such as
 * threads. In the example above, try-with-resources is used, which automatically calls close().
 *
 * <p>The surface of this class includes several types of Java methods for each of the API's
 * methods:
 *
 * <ol>
 *   <li>A "flattened" method. With this type of method, the fields of the request type have been
 *       converted into function parameters. It may be the case that not all fields are available as
 *       parameters, and not every API method will have a flattened method entry point.
 *   <li>A "request object" method. This type of method only takes one parameter, a request object,
 *       which must be constructed before the call. Not every API method will have a request object
 *       method.
 *   <li>A "callable" method. This type of method takes no parameters and returns an immutable API
 *       callable object, which can be used to initiate calls to the service.
 * </ol>
 *
 * <p>See the individual methods for example code.
 *
 * <p>Many parameters require resource names to be formatted in a particular way. To assist with
 * these names, this class includes a format method for each type of name, and additionally a parse
 * method to extract the individual identifiers contained within names that are returned.
 *
 * <p>This class can be customized by passing in a custom instance of AutoscalersSettings to
 * create(). For example:
 *
 * <p>To customize credentials:
 *
 * <pre>{@code
 * AutoscalersSettings autoscalersSettings =
 *     AutoscalersSettings.newBuilder()
 *         .setCredentialsProvider(FixedCredentialsProvider.create(myCredentials))
 *         .build();
 * AutoscalersClient autoscalersClient = AutoscalersClient.create(autoscalersSettings);
 * }</pre>
 *
 * <p>To customize the endpoint:
 *
 * <pre>{@code
 * AutoscalersSettings autoscalersSettings =
 *     AutoscalersSettings.newBuilder().setEndpoint(myEndpoint).build();
 * AutoscalersClient autoscalersClient = AutoscalersClient.create(autoscalersSettings);
 * }</pre>
 *
 * <p>Please refer to the GitHub repository's samples for more quickstart code snippets.
 */
@Generated("by gapic-generator-java")
public class AutoscalersClient implements BackgroundResource {
  private final AutoscalersSettings settings;
  private final AutoscalersStub stub;

  /** Constructs an instance of AutoscalersClient with default settings. */
  public static final AutoscalersClient create() throws IOException {
    return create(AutoscalersSettings.newBuilder().build());
  }

  /**
   * Constructs an instance of AutoscalersClient, using the given settings. The channels are created
   * based on the settings passed in, or defaults for any settings that are not set.
   */
  public static final AutoscalersClient create(AutoscalersSettings settings) throws IOException {
    return new AutoscalersClient(settings);
  }

  /**
   * Constructs an instance of AutoscalersClient, using the given stub for making calls. This is for
   * advanced usage - prefer using create(AutoscalersSettings).
   */
  @BetaApi("A restructuring of stub classes is planned, so this may break in the future")
  public static final AutoscalersClient create(AutoscalersStub stub) {
    return new AutoscalersClient(stub);
  }

  /**
   * Constructs an instance of AutoscalersClient, using the given settings. This is protected so
   * that it is easy to make a subclass, but otherwise, the static factory methods should be
   * preferred.
   */
  protected AutoscalersClient(AutoscalersSettings settings) throws IOException {
    this.settings = settings;
    this.stub = ((AutoscalersStubSettings) settings.getStubSettings()).createStub();
  }

  @BetaApi("A restructuring of stub classes is planned, so this may break in the future")
  protected AutoscalersClient(AutoscalersStub stub) {
    this.settings = null;
    this.stub = stub;
  }

  public final AutoscalersSettings getSettings() {
    return settings;
  }

  @BetaApi("A restructuring of stub classes is planned, so this may break in the future")
  public AutoscalersStub getStub() {
    return stub;
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves an aggregated list of autoscalers.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   String project = "project-309310695";
   *   for (Map.Entry<String, AutoscalersScopedList> element :
   *       autoscalersClient.aggregatedList(project).iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   *
   * @param project Project ID for this request.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final AggregatedListPagedResponse aggregatedList(String project) {
    AggregatedListAutoscalersRequest request =
        AggregatedListAutoscalersRequest.newBuilder().setProject(project).build();
    return aggregatedList(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves an aggregated list of autoscalers.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   AggregatedListAutoscalersRequest request =
   *       AggregatedListAutoscalersRequest.newBuilder()
   *           .setFilter("filter-1274492040")
   *           .setIncludeAllScopes(true)
   *           .setMaxResults(1128457243)
   *           .setOrderBy("orderBy-1207110587")
   *           .setPageToken("pageToken873572522")
   *           .setProject("project-309310695")
   *           .setReturnPartialSuccess(true)
   *           .build();
   *   for (Map.Entry<String, AutoscalersScopedList> element :
   *       autoscalersClient.aggregatedList(request).iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final AggregatedListPagedResponse aggregatedList(
      AggregatedListAutoscalersRequest request) {
    return aggregatedListPagedCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves an aggregated list of autoscalers.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   AggregatedListAutoscalersRequest request =
   *       AggregatedListAutoscalersRequest.newBuilder()
   *           .setFilter("filter-1274492040")
   *           .setIncludeAllScopes(true)
   *           .setMaxResults(1128457243)
   *           .setOrderBy("orderBy-1207110587")
   *           .setPageToken("pageToken873572522")
   *           .setProject("project-309310695")
   *           .setReturnPartialSuccess(true)
   *           .build();
   *   ApiFuture<Map.Entry<String, AutoscalersScopedList>> future =
   *       autoscalersClient.aggregatedListPagedCallable().futureCall(request);
   *   // Do something.
   *   for (Map.Entry<String, AutoscalersScopedList> element : future.get().iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   */
  public final UnaryCallable<AggregatedListAutoscalersRequest, AggregatedListPagedResponse>
      aggregatedListPagedCallable() {
    return stub.aggregatedListPagedCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves an aggregated list of autoscalers.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   AggregatedListAutoscalersRequest request =
   *       AggregatedListAutoscalersRequest.newBuilder()
   *           .setFilter("filter-1274492040")
   *           .setIncludeAllScopes(true)
   *           .setMaxResults(1128457243)
   *           .setOrderBy("orderBy-1207110587")
   *           .setPageToken("pageToken873572522")
   *           .setProject("project-309310695")
   *           .setReturnPartialSuccess(true)
   *           .build();
   *   while (true) {
   *     AutoscalerAggregatedList response =
   *         autoscalersClient.aggregatedListCallable().call(request);
   *     for (Map.Entry<String, AutoscalersScopedList> element : response.getResponsesList()) {
   *       // doThingsWith(element);
   *     }
   *     String nextPageToken = response.getNextPageToken();
   *     if (!Strings.isNullOrEmpty(nextPageToken)) {
   *       request = request.toBuilder().setPageToken(nextPageToken).build();
   *     } else {
   *       break;
   *     }
   *   }
   * }
   * }</pre>
   */
  public final UnaryCallable<AggregatedListAutoscalersRequest, AutoscalerAggregatedList>
      aggregatedListCallable() {
    return stub.aggregatedListCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Deletes the specified autoscaler.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   String project = "project-309310695";
   *   String zone = "zone3744684";
   *   String autoscaler = "autoscaler517258967";
   *   Operation response = autoscalersClient.deleteAsync(project, zone, autoscaler).get();
   * }
   * }</pre>
   *
   * @param project Project ID for this request.
   * @param zone Name of the zone for this request.
   * @param autoscaler Name of the autoscaler to delete.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final OperationFuture<Operation, Operation> deleteAsync(
      String project, String zone, String autoscaler) {
    DeleteAutoscalerRequest request =
        DeleteAutoscalerRequest.newBuilder()
            .setProject(project)
            .setZone(zone)
            .setAutoscaler(autoscaler)
            .build();
    return deleteAsync(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Deletes the specified autoscaler.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   DeleteAutoscalerRequest request =
   *       DeleteAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   Operation response = autoscalersClient.deleteAsync(request).get();
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  @BetaApi(
      "The surface for long-running operations is not stable yet and may change in the future.")
  public final OperationFuture<Operation, Operation> deleteAsync(DeleteAutoscalerRequest request) {
    return deleteOperationCallable().futureCall(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Deletes the specified autoscaler.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   DeleteAutoscalerRequest request =
   *       DeleteAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   OperationFuture<Operation, Operation> future =
   *       autoscalersClient.deleteOperationCallable().futureCall(request);
   *   // Do something.
   *   Operation response = future.get();
   * }
   * }</pre>
   */
  public final OperationCallable<DeleteAutoscalerRequest, Operation, Operation>
      deleteOperationCallable() {
    return stub.deleteOperationCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Deletes the specified autoscaler.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   DeleteAutoscalerRequest request =
   *       DeleteAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   ApiFuture<Operation> future = autoscalersClient.deleteCallable().futureCall(request);
   *   // Do something.
   *   Operation response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<DeleteAutoscalerRequest, Operation> deleteCallable() {
    return stub.deleteCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Returns the specified autoscaler resource. Gets a list of available autoscalers by making a
   * list() request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   String project = "project-309310695";
   *   String zone = "zone3744684";
   *   String autoscaler = "autoscaler517258967";
   *   Autoscaler response = autoscalersClient.get(project, zone, autoscaler);
   * }
   * }</pre>
   *
   * @param project Project ID for this request.
   * @param zone Name of the zone for this request.
   * @param autoscaler Name of the autoscaler to return.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final Autoscaler get(String project, String zone, String autoscaler) {
    GetAutoscalerRequest request =
        GetAutoscalerRequest.newBuilder()
            .setProject(project)
            .setZone(zone)
            .setAutoscaler(autoscaler)
            .build();
    return get(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Returns the specified autoscaler resource. Gets a list of available autoscalers by making a
   * list() request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   GetAutoscalerRequest request =
   *       GetAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setProject("project-309310695")
   *           .setZone("zone3744684")
   *           .build();
   *   Autoscaler response = autoscalersClient.get(request);
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final Autoscaler get(GetAutoscalerRequest request) {
    return getCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Returns the specified autoscaler resource. Gets a list of available autoscalers by making a
   * list() request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   GetAutoscalerRequest request =
   *       GetAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setProject("project-309310695")
   *           .setZone("zone3744684")
   *           .build();
   *   ApiFuture<Autoscaler> future = autoscalersClient.getCallable().futureCall(request);
   *   // Do something.
   *   Autoscaler response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<GetAutoscalerRequest, Autoscaler> getCallable() {
    return stub.getCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Creates an autoscaler in the specified project using the data included in the request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   String project = "project-309310695";
   *   String zone = "zone3744684";
   *   Autoscaler autoscalerResource = Autoscaler.newBuilder().build();
   *   Operation response = autoscalersClient.insertAsync(project, zone, autoscalerResource).get();
   * }
   * }</pre>
   *
   * @param project Project ID for this request.
   * @param zone Name of the zone for this request.
   * @param autoscalerResource The body resource for this request
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final OperationFuture<Operation, Operation> insertAsync(
      String project, String zone, Autoscaler autoscalerResource) {
    InsertAutoscalerRequest request =
        InsertAutoscalerRequest.newBuilder()
            .setProject(project)
            .setZone(zone)
            .setAutoscalerResource(autoscalerResource)
            .build();
    return insertAsync(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Creates an autoscaler in the specified project using the data included in the request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   InsertAutoscalerRequest request =
   *       InsertAutoscalerRequest.newBuilder()
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   Operation response = autoscalersClient.insertAsync(request).get();
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  @BetaApi(
      "The surface for long-running operations is not stable yet and may change in the future.")
  public final OperationFuture<Operation, Operation> insertAsync(InsertAutoscalerRequest request) {
    return insertOperationCallable().futureCall(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Creates an autoscaler in the specified project using the data included in the request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   InsertAutoscalerRequest request =
   *       InsertAutoscalerRequest.newBuilder()
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   OperationFuture<Operation, Operation> future =
   *       autoscalersClient.insertOperationCallable().futureCall(request);
   *   // Do something.
   *   Operation response = future.get();
   * }
   * }</pre>
   */
  public final OperationCallable<InsertAutoscalerRequest, Operation, Operation>
      insertOperationCallable() {
    return stub.insertOperationCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Creates an autoscaler in the specified project using the data included in the request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   InsertAutoscalerRequest request =
   *       InsertAutoscalerRequest.newBuilder()
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   ApiFuture<Operation> future = autoscalersClient.insertCallable().futureCall(request);
   *   // Do something.
   *   Operation response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<InsertAutoscalerRequest, Operation> insertCallable() {
    return stub.insertCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves a list of autoscalers contained within the specified zone.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   String project = "project-309310695";
   *   String zone = "zone3744684";
   *   for (Autoscaler element : autoscalersClient.list(project, zone).iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   *
   * @param project Project ID for this request.
   * @param zone Name of the zone for this request.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final ListPagedResponse list(String project, String zone) {
    ListAutoscalersRequest request =
        ListAutoscalersRequest.newBuilder().setProject(project).setZone(zone).build();
    return list(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves a list of autoscalers contained within the specified zone.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   ListAutoscalersRequest request =
   *       ListAutoscalersRequest.newBuilder()
   *           .setFilter("filter-1274492040")
   *           .setMaxResults(1128457243)
   *           .setOrderBy("orderBy-1207110587")
   *           .setPageToken("pageToken873572522")
   *           .setProject("project-309310695")
   *           .setReturnPartialSuccess(true)
   *           .setZone("zone3744684")
   *           .build();
   *   for (Autoscaler element : autoscalersClient.list(request).iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final ListPagedResponse list(ListAutoscalersRequest request) {
    return listPagedCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves a list of autoscalers contained within the specified zone.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   ListAutoscalersRequest request =
   *       ListAutoscalersRequest.newBuilder()
   *           .setFilter("filter-1274492040")
   *           .setMaxResults(1128457243)
   *           .setOrderBy("orderBy-1207110587")
   *           .setPageToken("pageToken873572522")
   *           .setProject("project-309310695")
   *           .setReturnPartialSuccess(true)
   *           .setZone("zone3744684")
   *           .build();
   *   ApiFuture<Autoscaler> future = autoscalersClient.listPagedCallable().futureCall(request);
   *   // Do something.
   *   for (Autoscaler element : future.get().iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   */
  public final UnaryCallable<ListAutoscalersRequest, ListPagedResponse> listPagedCallable() {
    return stub.listPagedCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves a list of autoscalers contained within the specified zone.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   ListAutoscalersRequest request =
   *       ListAutoscalersRequest.newBuilder()
   *           .setFilter("filter-1274492040")
   *           .setMaxResults(1128457243)
   *           .setOrderBy("orderBy-1207110587")
   *           .setPageToken("pageToken873572522")
   *           .setProject("project-309310695")
   *           .setReturnPartialSuccess(true)
   *           .setZone("zone3744684")
   *           .build();
   *   while (true) {
   *     AutoscalerList response = autoscalersClient.listCallable().call(request);
   *     for (Autoscaler element : response.getResponsesList()) {
   *       // doThingsWith(element);
   *     }
   *     String nextPageToken = response.getNextPageToken();
   *     if (!Strings.isNullOrEmpty(nextPageToken)) {
   *       request = request.toBuilder().setPageToken(nextPageToken).build();
   *     } else {
   *       break;
   *     }
   *   }
   * }
   * }</pre>
   */
  public final UnaryCallable<ListAutoscalersRequest, AutoscalerList> listCallable() {
    return stub.listCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates an autoscaler in the specified project using the data included in the request. This
   * method supports PATCH semantics and uses the JSON merge patch format and processing rules.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   String project = "project-309310695";
   *   String zone = "zone3744684";
   *   Autoscaler autoscalerResource = Autoscaler.newBuilder().build();
   *   Operation response = autoscalersClient.patchAsync(project, zone, autoscalerResource).get();
   * }
   * }</pre>
   *
   * @param project Project ID for this request.
   * @param zone Name of the zone for this request.
   * @param autoscalerResource The body resource for this request
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final OperationFuture<Operation, Operation> patchAsync(
      String project, String zone, Autoscaler autoscalerResource) {
    PatchAutoscalerRequest request =
        PatchAutoscalerRequest.newBuilder()
            .setProject(project)
            .setZone(zone)
            .setAutoscalerResource(autoscalerResource)
            .build();
    return patchAsync(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates an autoscaler in the specified project using the data included in the request. This
   * method supports PATCH semantics and uses the JSON merge patch format and processing rules.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   PatchAutoscalerRequest request =
   *       PatchAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   Operation response = autoscalersClient.patchAsync(request).get();
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  @BetaApi(
      "The surface for long-running operations is not stable yet and may change in the future.")
  public final OperationFuture<Operation, Operation> patchAsync(PatchAutoscalerRequest request) {
    return patchOperationCallable().futureCall(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates an autoscaler in the specified project using the data included in the request. This
   * method supports PATCH semantics and uses the JSON merge patch format and processing rules.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   PatchAutoscalerRequest request =
   *       PatchAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   OperationFuture<Operation, Operation> future =
   *       autoscalersClient.patchOperationCallable().futureCall(request);
   *   // Do something.
   *   Operation response = future.get();
   * }
   * }</pre>
   */
  public final OperationCallable<PatchAutoscalerRequest, Operation, Operation>
      patchOperationCallable() {
    return stub.patchOperationCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates an autoscaler in the specified project using the data included in the request. This
   * method supports PATCH semantics and uses the JSON merge patch format and processing rules.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   PatchAutoscalerRequest request =
   *       PatchAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   ApiFuture<Operation> future = autoscalersClient.patchCallable().futureCall(request);
   *   // Do something.
   *   Operation response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<PatchAutoscalerRequest, Operation> patchCallable() {
    return stub.patchCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates an autoscaler in the specified project using the data included in the request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   String project = "project-309310695";
   *   String zone = "zone3744684";
   *   Autoscaler autoscalerResource = Autoscaler.newBuilder().build();
   *   Operation response = autoscalersClient.updateAsync(project, zone, autoscalerResource).get();
   * }
   * }</pre>
   *
   * @param project Project ID for this request.
   * @param zone Name of the zone for this request.
   * @param autoscalerResource The body resource for this request
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final OperationFuture<Operation, Operation> updateAsync(
      String project, String zone, Autoscaler autoscalerResource) {
    UpdateAutoscalerRequest request =
        UpdateAutoscalerRequest.newBuilder()
            .setProject(project)
            .setZone(zone)
            .setAutoscalerResource(autoscalerResource)
            .build();
    return updateAsync(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates an autoscaler in the specified project using the data included in the request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   UpdateAutoscalerRequest request =
   *       UpdateAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   Operation response = autoscalersClient.updateAsync(request).get();
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  @BetaApi(
      "The surface for long-running operations is not stable yet and may change in the future.")
  public final OperationFuture<Operation, Operation> updateAsync(UpdateAutoscalerRequest request) {
    return updateOperationCallable().futureCall(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates an autoscaler in the specified project using the data included in the request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   UpdateAutoscalerRequest request =
   *       UpdateAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   OperationFuture<Operation, Operation> future =
   *       autoscalersClient.updateOperationCallable().futureCall(request);
   *   // Do something.
   *   Operation response = future.get();
   * }
   * }</pre>
   */
  public final OperationCallable<UpdateAutoscalerRequest, Operation, Operation>
      updateOperationCallable() {
    return stub.updateOperationCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates an autoscaler in the specified project using the data included in the request.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * try (AutoscalersClient autoscalersClient = AutoscalersClient.create()) {
   *   UpdateAutoscalerRequest request =
   *       UpdateAutoscalerRequest.newBuilder()
   *           .setAutoscaler("autoscaler517258967")
   *           .setAutoscalerResource(Autoscaler.newBuilder().build())
   *           .setProject("project-309310695")
   *           .setRequestId("requestId693933066")
   *           .setZone("zone3744684")
   *           .build();
   *   ApiFuture<Operation> future = autoscalersClient.updateCallable().futureCall(request);
   *   // Do something.
   *   Operation response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<UpdateAutoscalerRequest, Operation> updateCallable() {
    return stub.updateCallable();
  }

  @Override
  public final void close() {
    stub.close();
  }

  @Override
  public void shutdown() {
    stub.shutdown();
  }

  @Override
  public boolean isShutdown() {
    return stub.isShutdown();
  }

  @Override
  public boolean isTerminated() {
    return stub.isTerminated();
  }

  @Override
  public void shutdownNow() {
    stub.shutdownNow();
  }

  @Override
  public boolean awaitTermination(long duration, TimeUnit unit) throws InterruptedException {
    return stub.awaitTermination(duration, unit);
  }

  public static class AggregatedListPagedResponse
      extends AbstractPagedListResponse<
          AggregatedListAutoscalersRequest,
          AutoscalerAggregatedList,
          Map.Entry<String, AutoscalersScopedList>,
          AggregatedListPage,
          AggregatedListFixedSizeCollection> {

    public static ApiFuture<AggregatedListPagedResponse> createAsync(
        PageContext<
                AggregatedListAutoscalersRequest,
                AutoscalerAggregatedList,
                Map.Entry<String, AutoscalersScopedList>>
            context,
        ApiFuture<AutoscalerAggregatedList> futureResponse) {
      ApiFuture<AggregatedListPage> futurePage =
          AggregatedListPage.createEmptyPage().createPageAsync(context, futureResponse);
      return ApiFutures.transform(
          futurePage,
          input -> new AggregatedListPagedResponse(input),
          MoreExecutors.directExecutor());
    }

    private AggregatedListPagedResponse(AggregatedListPage page) {
      super(page, AggregatedListFixedSizeCollection.createEmptyCollection());
    }
  }

  public static class AggregatedListPage
      extends AbstractPage<
          AggregatedListAutoscalersRequest,
          AutoscalerAggregatedList,
          Map.Entry<String, AutoscalersScopedList>,
          AggregatedListPage> {

    private AggregatedListPage(
        PageContext<
                AggregatedListAutoscalersRequest,
                AutoscalerAggregatedList,
                Map.Entry<String, AutoscalersScopedList>>
            context,
        AutoscalerAggregatedList response) {
      super(context, response);
    }

    private static AggregatedListPage createEmptyPage() {
      return new AggregatedListPage(null, null);
    }

    @Override
    protected AggregatedListPage createPage(
        PageContext<
                AggregatedListAutoscalersRequest,
                AutoscalerAggregatedList,
                Map.Entry<String, AutoscalersScopedList>>
            context,
        AutoscalerAggregatedList response) {
      return new AggregatedListPage(context, response);
    }

    @Override
    public ApiFuture<AggregatedListPage> createPageAsync(
        PageContext<
                AggregatedListAutoscalersRequest,
                AutoscalerAggregatedList,
                Map.Entry<String, AutoscalersScopedList>>
            context,
        ApiFuture<AutoscalerAggregatedList> futureResponse) {
      return super.createPageAsync(context, futureResponse);
    }
  }

  public static class AggregatedListFixedSizeCollection
      extends AbstractFixedSizeCollection<
          AggregatedListAutoscalersRequest,
          AutoscalerAggregatedList,
          Map.Entry<String, AutoscalersScopedList>,
          AggregatedListPage,
          AggregatedListFixedSizeCollection> {

    private AggregatedListFixedSizeCollection(List<AggregatedListPage> pages, int collectionSize) {
      super(pages, collectionSize);
    }

    private static AggregatedListFixedSizeCollection createEmptyCollection() {
      return new AggregatedListFixedSizeCollection(null, 0);
    }

    @Override
    protected AggregatedListFixedSizeCollection createCollection(
        List<AggregatedListPage> pages, int collectionSize) {
      return new AggregatedListFixedSizeCollection(pages, collectionSize);
    }
  }

  public static class ListPagedResponse
      extends AbstractPagedListResponse<
          ListAutoscalersRequest, AutoscalerList, Autoscaler, ListPage, ListFixedSizeCollection> {

    public static ApiFuture<ListPagedResponse> createAsync(
        PageContext<ListAutoscalersRequest, AutoscalerList, Autoscaler> context,
        ApiFuture<AutoscalerList> futureResponse) {
      ApiFuture<ListPage> futurePage =
          ListPage.createEmptyPage().createPageAsync(context, futureResponse);
      return ApiFutures.transform(
          futurePage, input -> new ListPagedResponse(input), MoreExecutors.directExecutor());
    }

    private ListPagedResponse(ListPage page) {
      super(page, ListFixedSizeCollection.createEmptyCollection());
    }
  }

  public static class ListPage
      extends AbstractPage<ListAutoscalersRequest, AutoscalerList, Autoscaler, ListPage> {

    private ListPage(
        PageContext<ListAutoscalersRequest, AutoscalerList, Autoscaler> context,
        AutoscalerList response) {
      super(context, response);
    }

    private static ListPage createEmptyPage() {
      return new ListPage(null, null);
    }

    @Override
    protected ListPage createPage(
        PageContext<ListAutoscalersRequest, AutoscalerList, Autoscaler> context,
        AutoscalerList response) {
      return new ListPage(context, response);
    }

    @Override
    public ApiFuture<ListPage> createPageAsync(
        PageContext<ListAutoscalersRequest, AutoscalerList, Autoscaler> context,
        ApiFuture<AutoscalerList> futureResponse) {
      return super.createPageAsync(context, futureResponse);
    }
  }

  public static class ListFixedSizeCollection
      extends AbstractFixedSizeCollection<
          ListAutoscalersRequest, AutoscalerList, Autoscaler, ListPage, ListFixedSizeCollection> {

    private ListFixedSizeCollection(List<ListPage> pages, int collectionSize) {
      super(pages, collectionSize);
    }

    private static ListFixedSizeCollection createEmptyCollection() {
      return new ListFixedSizeCollection(null, 0);
    }

    @Override
    protected ListFixedSizeCollection createCollection(List<ListPage> pages, int collectionSize) {
      return new ListFixedSizeCollection(pages, collectionSize);
    }
  }
}
