/*
 * Copyright 2022 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.google.cloud.datastore;

import com.google.api.core.ObsoleteApi;
import com.google.common.base.MoreObjects;
import com.google.common.base.MoreObjects.ToStringHelper;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;

/** Represents a result of an {@link AggregationQuery} query submission. */
public class AggregationResult {

  private final Map<String, ? extends Value<?>> properties;

  public AggregationResult(Map<String, ? extends Value<?>> properties) {
    this.properties = properties;
  }

  /**
   * Returns a result value for the given alias. {@link #getLong(String)} is preferred over this
   * method, Use {@link #getLong(String)} wherever possible.
   *
   * @param alias A custom alias provided in the query or an autogenerated alias in the form of
   *     'property_\d'
   * @return An aggregation result value for the given alias.
   */
  @ObsoleteApi("Please use getLong(String) instead, see Github issue #1175 for details.")
  public Long get(String alias) {
    return getLong(alias);
  }

  /**
   * Returns a result value for the given alias.
   *
   * @param alias A custom alias provided in the query or an autogenerated alias in the form of
   *     'property_\d'
   * @return An aggregation result value for the given alias.
   */
  public Long getLong(String alias) {
    Value<?> value = properties.get(alias);
    switch (value.getType()) {
      case DOUBLE:
        return ((Double) value.get()).longValue();
      case LONG:
        return (Long) value.get();
      default:
        throw new RuntimeException(
            String.format("Unsupported type %s received for alias '%s'.", value.getType(), alias));
    }
  }

  /**
   * Returns a result value for the given alias.
   *
   * @param alias A custom alias provided in the query or an autogenerated alias in the form of
   *     'property_\d'
   * @return An aggregation result value for the given alias.
   */
  public Double getDouble(String alias) {
    Value<?> value = properties.get(alias);
    switch (value.getType()) {
      case LONG:
        return ((Long) value.get()).doubleValue();
      case DOUBLE:
        return (Double) value.get();
      default:
        throw new RuntimeException(
            String.format("Unsupported type %s received for alias '%s'.", value.getType(), alias));
    }
  }

  /**
   * Returns a result value for the given alias.
   *
   * @param alias A custom alias provided in the query or an autogenerated alias in the form of
   *     'property_\d'
   * @return An aggregation result value as a {@link Number} for the given alias.
   */
  public Number getNumber(String alias) {
    Value<?> value = properties.get(alias);
    switch (value.getType()) {
      case LONG:
        return (Long) value.get();
      case DOUBLE:
        return (Double) value.get();
      default:
        throw new RuntimeException(
            String.format("Unsupported type %s received for alias '%s'.", value.getType(), alias));
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    AggregationResult that = (AggregationResult) o;
    return properties.equals(that.properties);
  }

  @Override
  public int hashCode() {
    return Objects.hash(properties);
  }

  @Override
  public String toString() {
    ToStringHelper toStringHelper = MoreObjects.toStringHelper(this);
    for (Entry<String, ? extends Value<?>> entry : properties.entrySet()) {
      toStringHelper.add(entry.getKey(), entry.getValue().get());
    }
    return toStringHelper.toString();
  }
}
