/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner.spi.v1;

import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.grpc.CallOptions;
import io.grpc.Channel;
import io.grpc.ClientCall;
import io.grpc.ClientInterceptor;
import io.grpc.ForwardingClientCall;
import io.grpc.ForwardingClientCallListener;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

class WatchdogInterceptor
implements ClientInterceptor {
    private static final String PROPERTY_TIMEOUT_SECONDS = "com.google.cloud.spanner.watchdogTimeoutSeconds";
    private static final String PROPERTY_PERIOD_SECONDS = "com.google.cloud.spanner.watchdogPeriodSeconds";
    private static final int DEFAULT_TIMEOUT_SECONDS = 1800;
    private static final int DEFAULT_PERIOD_SECONDS = 10;
    private static final Logger logger = Logger.getLogger(WatchdogInterceptor.class.getName());
    private final long activityTimeoutNanos;
    private final TimeUnit activityTimeoutUnits;
    private final Ticker ticker;
    private final ConcurrentHashMap<MonitoredCall<?, ?>, MonitoredCall<?, ?>> monitoredCalls;

    WatchdogInterceptor(long activityTimeout, TimeUnit activityTimeoutUnits) {
        this(activityTimeout, activityTimeoutUnits, Ticker.systemTicker());
    }

    WatchdogInterceptor(long activityTimeout, TimeUnit activityTimeoutUnits, Ticker ticker) {
        Preconditions.checkArgument((activityTimeout > 0L ? 1 : 0) != 0, (Object)"activityTimeout must be positive");
        this.activityTimeoutNanos = activityTimeoutUnits.toNanos(activityTimeout);
        this.activityTimeoutUnits = (TimeUnit)((Object)Preconditions.checkNotNull((Object)((Object)activityTimeoutUnits)));
        this.ticker = (Ticker)Preconditions.checkNotNull((Object)ticker);
        this.monitoredCalls = new ConcurrentHashMap(128);
    }

    private static int systemProperty(String name, int defaultValue) {
        String stringValue = System.getProperty(name, "");
        return stringValue.isEmpty() ? defaultValue : Integer.parseInt(stringValue);
    }

    @Nullable
    static ClientInterceptor newDefaultWatchdogInterceptor() {
        int timeoutSeconds = WatchdogInterceptor.systemProperty(PROPERTY_TIMEOUT_SECONDS, 1800);
        if (timeoutSeconds <= 0) {
            return new ClientInterceptor(){

                public <ReqT, RespT> ClientCall<ReqT, RespT> interceptCall(MethodDescriptor<ReqT, RespT> methodDescriptor, CallOptions callOptions, Channel channel) {
                    return channel.newCall(methodDescriptor, callOptions);
                }
            };
        }
        int periodSeconds = WatchdogInterceptor.systemProperty(PROPERTY_PERIOD_SECONDS, 10);
        final WatchdogInterceptor interceptor = new WatchdogInterceptor(timeoutSeconds, TimeUnit.SECONDS);
        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("Cloud-Spanner-WatchdogInterceptor-%d").build());
        executor.scheduleWithFixedDelay(new Runnable(){

            @Override
            public void run() {
                interceptor.tick();
            }
        }, periodSeconds, periodSeconds, TimeUnit.SECONDS);
        logger.log(Level.FINE, "Created watchdog interceptor with activity timeout of {0}s and period {1}s", new Object[]{timeoutSeconds, periodSeconds});
        return interceptor;
    }

    void tick() {
        for (MonitoredCall call : this.monitoredCalls.keySet()) {
            call.checkActivity();
        }
    }

    public <ReqT, RespT> ClientCall<ReqT, RespT> interceptCall(MethodDescriptor<ReqT, RespT> methodDescriptor, CallOptions callOptions, Channel channel) {
        return new MonitoredCall(channel.newCall(methodDescriptor, callOptions));
    }

    private void registerCall(MonitoredCall<?, ?> call) {
        this.monitoredCalls.put(call, call);
    }

    private void unregisterCall(MonitoredCall<?, ?> call) {
        this.monitoredCalls.remove(call);
    }

    private class MonitoredCall<ReqT, RespT>
    extends ForwardingClientCall.SimpleForwardingClientCall<ReqT, RespT> {
        private volatile long lastActivityNanos;
        private volatile boolean stoppedByWatchdog;
        private final AtomicBoolean cancelled;

        MonitoredCall(ClientCall<ReqT, RespT> delegate) {
            super(delegate);
            this.cancelled = new AtomicBoolean(false);
        }

        public void start(ClientCall.Listener<RespT> responseListener, Metadata headers) {
            this.recordActivity();
            WatchdogInterceptor.this.registerCall(this);
            ForwardingClientCallListener.SimpleForwardingClientCallListener listener = new ForwardingClientCallListener.SimpleForwardingClientCallListener<RespT>(responseListener){

                public void onHeaders(Metadata headers) {
                    MonitoredCall.this.recordActivity();
                    super.onHeaders(headers);
                }

                public void onMessage(RespT message) {
                    MonitoredCall.this.recordActivity();
                    super.onMessage(message);
                }

                public void onReady() {
                    MonitoredCall.this.recordActivity();
                    super.onReady();
                }

                public void onClose(Status status, Metadata trailers) {
                    WatchdogInterceptor.this.unregisterCall(MonitoredCall.this);
                    super.onClose(MonitoredCall.this.handleStatus(status), trailers);
                }
            };
            super.start((ClientCall.Listener)listener, headers);
        }

        void recordActivity() {
            this.lastActivityNanos = WatchdogInterceptor.this.ticker.read();
        }

        void checkActivity() {
            if (WatchdogInterceptor.this.ticker.read() - this.lastActivityNanos > WatchdogInterceptor.this.activityTimeoutNanos && this.cancelled.compareAndSet(false, true)) {
                this.stoppedByWatchdog = true;
                this.delegate().cancel("Cancelled by activity watchdog", null);
                logger.log(Level.WARNING, "Cancelled due to exceeding inactivity timeout of {0} {1}", new Object[]{WatchdogInterceptor.this.activityTimeoutUnits.convert(WatchdogInterceptor.this.activityTimeoutNanos, TimeUnit.NANOSECONDS), WatchdogInterceptor.this.activityTimeoutUnits});
            }
        }

        Status handleStatus(Status status) {
            if (this.stoppedByWatchdog && status.getCode() == Status.Code.CANCELLED) {
                return Status.UNAVAILABLE.withDescription("Aborted by RPC activity watchdog [timeout=" + WatchdogInterceptor.this.activityTimeoutUnits.convert(WatchdogInterceptor.this.activityTimeoutNanos, TimeUnit.NANOSECONDS) + " " + (Object)((Object)WatchdogInterceptor.this.activityTimeoutUnits) + "]");
            }
            return status;
        }

        public void cancel(@Nullable String message, @Nullable Throwable cause) {
            if (this.cancelled.compareAndSet(false, true)) {
                super.cancel(message, cause);
            }
        }
    }
}

