/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner.it.slow;

import com.google.api.client.util.Lists;
import com.google.api.gax.grpc.GrpcInterceptorProvider;
import com.google.api.gax.longrunning.OperationFuture;
import com.google.api.gax.paging.Page;
import com.google.api.gax.rpc.FailedPreconditionException;
import com.google.cloud.Timestamp;
import com.google.cloud.spanner.Backup;
import com.google.cloud.spanner.BackupId;
import com.google.cloud.spanner.BackupInfo;
import com.google.cloud.spanner.Database;
import com.google.cloud.spanner.DatabaseAdminClient;
import com.google.cloud.spanner.DatabaseClient;
import com.google.cloud.spanner.DatabaseId;
import com.google.cloud.spanner.Dialect;
import com.google.cloud.spanner.ErrorCode;
import com.google.cloud.spanner.Instance;
import com.google.cloud.spanner.InstanceAdminClient;
import com.google.cloud.spanner.InstanceId;
import com.google.cloud.spanner.IntegrationTestEnv;
import com.google.cloud.spanner.Mutation;
import com.google.cloud.spanner.Options;
import com.google.cloud.spanner.Restore;
import com.google.cloud.spanner.ResultSet;
import com.google.cloud.spanner.SlowTest;
import com.google.cloud.spanner.Spanner;
import com.google.cloud.spanner.SpannerException;
import com.google.cloud.spanner.SpannerOptions;
import com.google.cloud.spanner.Statement;
import com.google.cloud.spanner.encryption.BackupEncryptionConfig;
import com.google.cloud.spanner.encryption.EncryptionConfigs;
import com.google.cloud.spanner.encryption.RestoreEncryptionConfig;
import com.google.cloud.spanner.testing.EmulatorSpannerHelper;
import com.google.cloud.spanner.testing.RemoteSpannerHelper;
import com.google.cloud.spanner.testing.TimestampHelper;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Iterables;
import com.google.longrunning.Operation;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.spanner.admin.database.v1.CreateBackupMetadata;
import com.google.spanner.admin.database.v1.RestoreDatabaseMetadata;
import com.google.spanner.admin.database.v1.RestoreSourceType;
import io.grpc.CallOptions;
import io.grpc.Channel;
import io.grpc.ClientCall;
import io.grpc.ClientInterceptor;
import io.grpc.ForwardingClientCall;
import io.grpc.ForwardingClientCallListener;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.StreamSupport;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.FixMethodOrder;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;
import org.junit.runners.MethodSorters;

@Category(value={SlowTest.class})
@RunWith(value=JUnit4.class)
@FixMethodOrder(value=MethodSorters.NAME_ASCENDING)
public class ITBackupTest {
    private static final long BACKUP_TIMEOUT_MINUTES = 30L;
    private static final long DATABASE_TIMEOUT_MINUTES = 5L;
    private static final Logger logger = Logger.getLogger(ITBackupTest.class.getName());
    private static final String EXPECTED_OP_NAME_FORMAT = "%s/backups/%s/operations/";
    private static final String KMS_KEY_NAME_PROPERTY = "spanner.testenv.kms_key.name";
    @ClassRule
    public static IntegrationTestEnv env = new IntegrationTestEnv();
    private static String keyName;
    private static DatabaseAdminClient dbAdminClient;
    private static Instance instance;
    private static RemoteSpannerHelper testHelper;
    private static final List<String> databases;
    private static final List<String> backups;
    private static String projectId;
    private static String instanceId;

    @BeforeClass
    public static void setup() {
        Assume.assumeFalse((String)"backups are not supported on the emulator", (boolean)EmulatorSpannerHelper.isUsingEmulator());
        keyName = System.getProperty(KMS_KEY_NAME_PROPERTY);
        Preconditions.checkNotNull((Object)keyName, (Object)"Key name is null, please set a key to be used for this test. The necessary permissions should be grant to the spanner service account according to the CMEK user guide.");
        logger.info("Setting up tests");
        testHelper = env.getTestHelper();
        dbAdminClient = testHelper.getClient().getDatabaseAdminClient();
        InstanceAdminClient instanceAdminClient = testHelper.getClient().getInstanceAdminClient();
        instance = instanceAdminClient.getInstance(testHelper.getInstanceId().getInstance());
        projectId = testHelper.getInstanceId().getProject();
        instanceId = testHelper.getInstanceId().getInstance();
        logger.info("Finished setup");
        logger.info("Cancelling long-running test backup operations");
        Pattern pattern = Pattern.compile(".*/backups/testbck_\\d{6}_\\d{4}_bck\\d/operations/.*");
        try {
            for (Operation operation : dbAdminClient.listBackupOperations(instance.getId().getInstance(), new Options.ListOption[0]).iterateAll()) {
                Matcher matcher = pattern.matcher(operation.getName());
                if (!matcher.matches() || operation.getDone()) continue;
                Timestamp currentTime = Timestamp.now();
                Timestamp startTime = Timestamp.fromProto((com.google.protobuf.Timestamp)((CreateBackupMetadata)operation.getMetadata().unpack(CreateBackupMetadata.class)).getProgress().getStartTime());
                long diffSeconds = currentTime.getSeconds() - startTime.getSeconds();
                if (TimeUnit.HOURS.convert(diffSeconds, TimeUnit.SECONDS) < 6L) continue;
                logger.warning(String.format("Cancelling test backup operation %s that was started at %s", operation.getName(), startTime));
                dbAdminClient.cancelOperation(operation.getName());
            }
        }
        catch (InvalidProtocolBufferException e) {
            logger.log(Level.WARNING, "Could not list all existing backup operations.", e);
        }
        logger.info("Finished checking existing test backup operations");
    }

    @AfterClass
    public static void tearDown() throws Exception {
        logger.info("Starting test teardown");
        for (String backup : backups) {
            logger.info(String.format("Waiting for optimize operation for backup %s to finish", backup));
            ITBackupTest.waitForDbOperations(backup);
            logger.info(String.format("Deleting backup %s", backup));
            dbAdminClient.deleteBackup(testHelper.getInstanceId().getInstance(), backup);
        }
        backups.clear();
        for (String db : databases) {
            logger.info(String.format("Dropping database %s", db));
            dbAdminClient.dropDatabase(testHelper.getInstanceId().getInstance(), db);
        }
    }

    private static void waitForDbOperations(String backupId) throws InterruptedException {
        try {
            Backup backupMetadata = dbAdminClient.getBackup(testHelper.getInstanceId().getInstance(), backupId);
            Assert.assertNotNull((Object)backupMetadata.getProto());
            boolean allDbOpsDone = false;
            while (!allDbOpsDone) {
                allDbOpsDone = true;
                block3: for (String referencingDb : backupMetadata.getProto().getReferencingDatabasesList()) {
                    String filter = String.format("name:%s/operations/ AND ((metadata.@type:type.googleapis.com/google.spanner.admin.database.v1.OptimizeRestoredDatabaseMetadata) OR (metadata.@type:type.googleapis.com/google.spanner.admin.database.v1.RestoreDatabaseMetadata))", referencingDb);
                    for (Operation op : dbAdminClient.listDatabaseOperations(testHelper.getInstanceId().getInstance(), new Options.ListOption[]{Options.filter((String)filter)}).iterateAll()) {
                        if (op.getDone()) continue;
                        Thread.sleep(5000L);
                        allDbOpsDone = false;
                        continue block3;
                    }
                }
            }
        }
        catch (SpannerException e) {
            if (e.getErrorCode() == ErrorCode.NOT_FOUND) {
                return;
            }
            throw e;
        }
    }

    @Test
    public void test01_Backups() throws InterruptedException, ExecutionException, TimeoutException {
        String databaseId = testHelper.getUniqueDatabaseId() + "_db1";
        Database sourceDatabase = dbAdminClient.newDatabaseBuilder(DatabaseId.of((String)projectId, (String)instanceId, (String)databaseId)).setEncryptionConfig(EncryptionConfigs.customerManagedEncryption((String)keyName)).build();
        logger.info(String.format("Creating test database %s", databaseId));
        OperationFuture createDatabaseOperation = dbAdminClient.createDatabase(sourceDatabase, Collections.singletonList("CREATE TABLE FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)"));
        Database database = (Database)createDatabaseOperation.get(5L, TimeUnit.MINUTES);
        databases.add(database.getId().getDatabase());
        DatabaseClient client = testHelper.getDatabaseClient(database);
        client.writeAtLeastOnce(Collections.singletonList(((Mutation.WriteBuilder)((Mutation.WriteBuilder)Mutation.newInsertOrUpdateBuilder((String)"FOO").set("ID").to(1L)).set("NAME").to("TEST")).build()));
        this.testDatabaseEncryption(database, keyName);
        this.testDatabaseDialect(database, Dialect.GOOGLE_STANDARD_SQL);
        String backupId = testHelper.getUniqueBackupId() + "_bck1";
        Timestamp expireTime = TimestampHelper.afterDays((int)7);
        Timestamp versionTime = this.getCurrentTimestamp(client);
        logger.info(String.format("Creating backup %s", backupId));
        Backup backupToCreate = dbAdminClient.newBackupBuilder(BackupId.of((String)projectId, (String)instanceId, (String)backupId)).setDatabase(database.getId()).setExpireTime(expireTime).setVersionTime(versionTime).setEncryptionConfig((BackupEncryptionConfig)EncryptionConfigs.customerManagedEncryption((String)keyName)).build();
        OperationFuture operation = dbAdminClient.createBackup(backupToCreate);
        backups.add(backupId);
        this.testMetadata((OperationFuture<Backup, CreateBackupMetadata>)operation, backupId, database);
        logger.info("Waiting for backup operation to finish");
        Backup backup = (Backup)operation.get(30L, TimeUnit.MINUTES);
        this.testBackupVersionTime(backup, versionTime);
        this.testBackupEncryption(backup, keyName);
        Timestamp commitTs = client.writeAtLeastOnce(Collections.singletonList(((Mutation.WriteBuilder)((Mutation.WriteBuilder)Mutation.newInsertOrUpdateBuilder((String)"FOO").set("ID").to(2L)).set("NAME").to("TEST2")).build()));
        logger.info("Listing all backups");
        Assert.assertTrue((boolean)Iterables.contains((Iterable)instance.listBackups(new Options.ListOption[0]).iterateAll(), (Object)backup));
        logger.info("Listing backups with name bck1");
        Assert.assertTrue((boolean)Iterables.elementsEqual((Iterable)dbAdminClient.listBackups(instanceId, new Options.ListOption[]{Options.filter((String)String.format("name:%s", backup.getId().getName()))}).iterateAll(), Collections.singleton(backup)));
        logger.info("Listing ready backups");
        Iterable readyBackups = dbAdminClient.listBackups(instanceId, new Options.ListOption[]{Options.filter((String)"state:READY")}).iterateAll();
        Assert.assertTrue((boolean)Iterables.contains((Iterable)readyBackups, (Object)backup));
        logger.info("Listing backups for database db1");
        Assert.assertTrue((boolean)Iterables.elementsEqual((Iterable)dbAdminClient.listBackups(instanceId, new Options.ListOption[]{Options.filter((String)String.format("database:%s", database.getId().getName()))}).iterateAll(), Collections.singleton(backup)));
        Timestamp ts = Timestamp.ofTimeSecondsAndNanos((long)commitTs.getSeconds(), (int)0);
        logger.info(String.format("Listing backups created before %s", ts));
        Assert.assertTrue((boolean)Iterables.contains((Iterable)dbAdminClient.listBackups(instanceId, new Options.ListOption[]{Options.filter((String)String.format("create_time<\"%s\"", ts))}).iterateAll(), (Object)backup));
        logger.info("Listing backups with size>0");
        Assert.assertTrue((boolean)Iterables.contains((Iterable)dbAdminClient.listBackups(instanceId, new Options.ListOption[]{Options.filter((String)"size_bytes>0")}).iterateAll(), (Object)backup));
        this.testPagination();
        logger.info("Finished listBackup tests");
        this.testGetBackup(database, backupId, expireTime);
        this.testUpdateBackup(backup);
        this.testCreateInvalidExpirationDate(database);
        this.testRestore(backup, versionTime, keyName);
        this.testCancelBackupOperation(database);
        logger.info("Finished all backup tests");
    }

    @Test
    public void test02_RetryNonIdempotentRpcsReturningLongRunningOperations() throws Exception {
        Timestamp initialDbCreateTime;
        String initialDatabaseId;
        Assume.assumeFalse((String)"Querying long-running operations is not supported on the emulator", (boolean)EmulatorSpannerHelper.isUsingEmulator());
        ArrayList<Database> databases = new ArrayList<Database>();
        InjectErrorInterceptorProvider createDbInterceptor = new InjectErrorInterceptorProvider("CreateDatabase");
        SpannerOptions options = testHelper.getOptions().toBuilder().setInterceptorProvider((GrpcInterceptorProvider)createDbInterceptor).build();
        try (Spanner spanner = (Spanner)options.getService();){
            initialDatabaseId = testHelper.getUniqueDatabaseId();
            DatabaseAdminClient client = spanner.getDatabaseAdminClient();
            OperationFuture op = client.createDatabase(testHelper.getInstanceId().getInstance(), initialDatabaseId, Collections.emptyList());
            databases.add((Database)op.get(5L, TimeUnit.MINUTES));
            initialDbCreateTime = ((Database)op.get(5L, TimeUnit.MINUTES)).getCreateTime();
            Assert.assertEquals((long)1L, (long)createDbInterceptor.methodCount.get());
            Assert.assertTrue((createDbInterceptor.getOperationCount.get() >= 1 ? 1 : 0) != 0);
        }
        InjectErrorInterceptorProvider createBackupInterceptor = new InjectErrorInterceptorProvider("CreateBackup");
        options = testHelper.getOptions().toBuilder().setInterceptorProvider((GrpcInterceptorProvider)createBackupInterceptor).build();
        String backupId = String.format("test-bck-%08d", new Random().nextInt(100000000));
        try (Spanner spanner = (Spanner)options.getService();){
            String databaseId = ((Database)databases.get(0)).getId().getDatabase();
            DatabaseAdminClient client = spanner.getDatabaseAdminClient();
            OperationFuture op = client.createBackup(testHelper.getInstanceId().getInstance(), backupId, databaseId, Timestamp.ofTimeSecondsAndNanos((long)(Timestamp.now().getSeconds() + TimeUnit.SECONDS.convert(7L, TimeUnit.DAYS)), (int)0));
            Stopwatch watch = Stopwatch.createStarted();
            while (createBackupInterceptor.methodCount.get() < 1 && createBackupInterceptor.getOperationCount.get() < 1 && watch.elapsed(TimeUnit.SECONDS) < 120L) {
                Thread.sleep(5000L);
            }
            client.cancelOperation(op.getName());
            Assert.assertEquals((long)1L, (long)createBackupInterceptor.methodCount.get());
            Assert.assertTrue((createBackupInterceptor.getOperationCount.get() >= 1 ? 1 : 0) != 0);
        }
        if (!backups.isEmpty()) {
            InjectErrorInterceptorProvider restoreBackupInterceptor = new InjectErrorInterceptorProvider("RestoreDatabase");
            options = testHelper.getOptions().toBuilder().setInterceptorProvider((GrpcInterceptorProvider)restoreBackupInterceptor).build();
            try (Spanner spanner = (Spanner)options.getService();){
                String restoredDbId = testHelper.getUniqueDatabaseId();
                DatabaseAdminClient client = spanner.getDatabaseAdminClient();
                OperationFuture op = client.restoreDatabase(testHelper.getInstanceId().getInstance(), backups.get(0), testHelper.getInstanceId().getInstance(), restoredDbId);
                Stopwatch watch = Stopwatch.createStarted();
                while (restoreBackupInterceptor.methodCount.get() < 1 && restoreBackupInterceptor.getOperationCount.get() < 1 && watch.elapsed(TimeUnit.SECONDS) < 120L) {
                    Thread.sleep(5000L);
                }
                try {
                    client.cancelOperation(op.getName());
                }
                catch (SpannerException | ExecutionException throwable) {
                    // empty catch block
                }
                Assert.assertEquals((long)1L, (long)restoreBackupInterceptor.methodCount.get());
                Assert.assertTrue((restoreBackupInterceptor.getOperationCount.get() >= 1 ? 1 : 0) != 0);
            }
        }
        createDbInterceptor = new InjectErrorInterceptorProvider("CreateDatabase");
        options = testHelper.getOptions().toBuilder().setInterceptorProvider((GrpcInterceptorProvider)createDbInterceptor).build();
        spanner = (Spanner)options.getService();
        try {
            DatabaseAdminClient client = spanner.getDatabaseAdminClient();
            client.dropDatabase(testHelper.getInstanceId().getInstance(), initialDatabaseId);
            OperationFuture op = client.createDatabase(testHelper.getInstanceId().getInstance(), initialDatabaseId, Collections.emptyList());
            Timestamp secondCreationTime = ((Database)op.get(5L, TimeUnit.MINUTES)).getCreateTime();
            Assert.assertTrue((secondCreationTime.compareTo(initialDbCreateTime) >= 0 ? 1 : 0) != 0);
            Assert.assertEquals((long)1L, (long)createDbInterceptor.methodCount.get());
            Assert.assertTrue((createDbInterceptor.getOperationCount.get() >= 1 ? 1 : 0) != 0);
        }
        finally {
            if (spanner != null) {
                spanner.close();
            }
        }
    }

    @Test
    public void test03_Delete() throws InterruptedException {
        Assert.assertFalse((String)"No backups created", (boolean)backups.isEmpty());
        String backupId = backups.get(0);
        ITBackupTest.waitForDbOperations(backupId);
        logger.info(String.format("Fetching backup %s", backupId));
        Backup backup = instance.getBackup(backupId);
        logger.info(String.format("Deleting backup %s", backupId));
        backup.delete();
        logger.info(String.format("Fetching non-existent backup %s", backupId));
        SpannerException exception = (SpannerException)Assert.assertThrows(SpannerException.class, () -> instance.getBackup(backupId));
        Assert.assertEquals((Object)ErrorCode.NOT_FOUND, (Object)exception.getErrorCode());
        logger.info(String.format("Deleting non-existent backup %s", backupId));
        backup.delete();
        logger.info("Finished delete tests");
    }

    @Test(expected=SpannerException.class)
    public void test04_backupCreationWithVersionTimeTooFarInThePastFails() throws Exception {
        Database testDatabase = testHelper.createTestDatabase(new String[0]);
        DatabaseId databaseId = testDatabase.getId();
        InstanceId instanceId = databaseId.getInstanceId();
        String backupId = testHelper.getUniqueBackupId();
        Timestamp expireTime = TimestampHelper.afterDays((int)7);
        Timestamp versionTime = TimestampHelper.daysAgo((int)30);
        Backup backupToCreate = dbAdminClient.newBackupBuilder(BackupId.of((InstanceId)instanceId, (String)backupId)).setDatabase(databaseId).setExpireTime(expireTime).setVersionTime(versionTime).build();
        this.getOrThrow(dbAdminClient.createBackup(backupToCreate));
    }

    @Test(expected=SpannerException.class)
    public void test05_backupCreationWithVersionTimeInTheFutureFails() throws Exception {
        Database testDatabase = testHelper.createTestDatabase(new String[0]);
        DatabaseId databaseId = testDatabase.getId();
        InstanceId instanceId = databaseId.getInstanceId();
        String backupId = testHelper.getUniqueBackupId();
        Timestamp expireTime = TimestampHelper.afterDays((int)7);
        Timestamp versionTime = TimestampHelper.afterDays((int)1);
        Backup backupToCreate = dbAdminClient.newBackupBuilder(BackupId.of((InstanceId)instanceId, (String)backupId)).setDatabase(databaseId).setExpireTime(expireTime).setVersionTime(versionTime).build();
        this.getOrThrow(dbAdminClient.createBackup(backupToCreate));
    }

    private <T> void getOrThrow(OperationFuture<T, ?> operation) throws InterruptedException, ExecutionException {
        try {
            operation.get();
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof SpannerException) {
                throw (SpannerException)e.getCause();
            }
            throw e;
        }
    }

    private Timestamp getCurrentTimestamp(DatabaseClient client) {
        try (ResultSet resultSet = client.singleUse().executeQuery(Statement.of((String)"SELECT CURRENT_TIMESTAMP()"), new Options.QueryOption[0]);){
            resultSet.next();
            Timestamp timestamp = resultSet.getTimestamp(0);
            return timestamp;
        }
    }

    private void testBackupVersionTime(Backup backup, Timestamp versionTime) {
        logger.info("Verifying backup version time for " + backup.getId());
        Assert.assertEquals((Object)versionTime, (Object)backup.getVersionTime());
        logger.info("Done verifying backup version time for " + backup.getId());
    }

    private void testDatabaseEncryption(Database database, String expectedKey) {
        logger.info("Verifying database encryption for " + database.getId());
        Assert.assertNotNull((Object)database.getEncryptionConfig());
        Assert.assertEquals((Object)expectedKey, (Object)database.getEncryptionConfig().getKmsKeyName());
        logger.info("Done verifying database encryption for " + database.getId());
    }

    private void testDatabaseDialect(Database database, Dialect expectedDialect) {
        logger.info("Verifying dialect for " + database.getId());
        Assert.assertNotNull((Object)database.getDialect());
        Assert.assertEquals((Object)expectedDialect, (Object)database.getDialect());
        logger.info("Done verifying database dialect for " + database.getId());
    }

    private void testBackupEncryption(Backup backup, String expectedKey) {
        logger.info("Verifying backup encryption for " + backup.getId());
        Assert.assertNotNull((Object)backup.getEncryptionInfo());
        Assert.assertTrue((boolean)backup.getEncryptionInfo().getKmsKeyVersion().contains(expectedKey));
        logger.info("Done verifying backup encryption for " + backup.getId());
    }

    private void testMetadata(OperationFuture<Backup, CreateBackupMetadata> operation, String backupId, Database database) throws InterruptedException, ExecutionException {
        logger.info("Getting operation metadata");
        CreateBackupMetadata metadata1 = (CreateBackupMetadata)operation.getMetadata().get();
        String expectedOperationName1 = String.format(EXPECTED_OP_NAME_FORMAT, testHelper.getInstanceId().getName(), backupId);
        Assert.assertTrue((boolean)operation.getName().startsWith(expectedOperationName1));
        Assert.assertEquals((Object)database.getId().getName(), (Object)metadata1.getDatabase());
        Assert.assertEquals((Object)BackupId.of((InstanceId)testHelper.getInstanceId(), (String)backupId).getName(), (Object)metadata1.getName());
        logger.info("Finished metadata tests");
    }

    private void testCreateInvalidExpirationDate(Database database) {
        Timestamp expireTime = TimestampHelper.daysAgo((int)1);
        String backupId = testHelper.getUniqueBackupId();
        logger.info(String.format("Creating backup %s with invalid expiration date", backupId));
        OperationFuture op = dbAdminClient.createBackup(instanceId, backupId, database.getId().getDatabase(), expireTime);
        backups.add(backupId);
        ExecutionException executionException = (ExecutionException)Assert.assertThrows(ExecutionException.class, () -> op.get());
        Throwable cause = executionException.getCause();
        Assert.assertEquals(SpannerException.class, cause.getClass());
        SpannerException spannerException = (SpannerException)cause;
        Assert.assertEquals((Object)ErrorCode.INVALID_ARGUMENT, (Object)spannerException.getErrorCode());
    }

    private void testCancelBackupOperation(Database database) throws InterruptedException, ExecutionException {
        Timestamp expireTime = TimestampHelper.afterDays((int)7);
        String backupId = testHelper.getUniqueBackupId();
        logger.info(String.format("Starting to create backup %s", backupId));
        OperationFuture op = dbAdminClient.createBackup(instanceId, backupId, database.getId().getDatabase(), expireTime);
        backups.add(backupId);
        logger.info(String.format("Cancelling the creation of backup %s", backupId));
        dbAdminClient.cancelOperation(op.getName());
        logger.info("Fetching backup operations");
        boolean operationFound = false;
        for (Operation operation : dbAdminClient.listBackupOperations(instanceId, new Options.ListOption[]{Options.filter((String)String.format("name:%s", op.getName()))}).iterateAll()) {
            Assert.assertEquals((long)Status.Code.CANCELLED.value(), (long)operation.getError().getCode());
            operationFound = true;
        }
        Assert.assertTrue((boolean)operationFound);
        logger.info("Finished cancel test");
    }

    private void testGetBackup(Database database, String backupId, Timestamp expireTime) {
        logger.info(String.format("Getting backup %s", backupId));
        Backup backup = instance.getBackup(backupId);
        Assert.assertEquals((Object)BackupInfo.State.READY, (Object)backup.getState());
        Assert.assertTrue((backup.getSize() > 0L ? 1 : 0) != 0);
        Assert.assertEquals((Object)expireTime, (Object)backup.getExpireTime());
        Assert.assertEquals((Object)database.getId(), (Object)backup.getDatabase());
    }

    private void testUpdateBackup(Backup backup) {
        Timestamp tomorrow = TimestampHelper.afterDays((int)1);
        backup = backup.toBuilder().setExpireTime(tomorrow).build();
        logger.info(String.format("Updating expire time of backup %s to 1 week", backup.getId().getBackup()));
        backup.updateExpireTime();
        logger.info(String.format("Reloading backup %s", backup.getId().getBackup()));
        backup = backup.reload();
        Assert.assertEquals((Object)tomorrow, (Object)backup.getExpireTime());
        Timestamp in5Minutes = TimestampHelper.afterMinutes((int)5);
        Backup backupWithNewExpireTime = backup.toBuilder().setExpireTime(in5Minutes).build();
        logger.info(String.format("Updating expire time of backup %s to 5 minutes", backup.getId().getBackup()));
        SpannerException spannerException = (SpannerException)Assert.assertThrows(SpannerException.class, () -> ((Backup)backupWithNewExpireTime).updateExpireTime());
        Assert.assertEquals((Object)ErrorCode.INVALID_ARGUMENT, (Object)spannerException.getErrorCode());
        backup = backup.reload();
        Assert.assertEquals((Object)tomorrow, (Object)backup.getExpireTime());
    }

    private void testPagination() {
        logger.info("Listing backups using pagination");
        ArrayList initialBackups = Lists.newArrayList((Iterable)dbAdminClient.listBackups(instanceId, new Options.ListOption[0]).iterateAll());
        int numBackups = 0;
        logger.info("Fetching first page");
        Page page = dbAdminClient.listBackups(instanceId, new Options.ListOption[]{Options.pageSize((int)1)});
        Assert.assertEquals((long)1L, (long)Iterables.size((Iterable)page.getValues()));
        ++numBackups;
        Assert.assertFalse((boolean)page.hasNextPage());
        HashSet<String> seenPageTokens = new HashSet<String>();
        seenPageTokens.add("");
        while (page.hasNextPage()) {
            logger.info(String.format("Fetching page %d with page token %s", numBackups + 1, page.getNextPageToken()));
            if (seenPageTokens.contains(page.getNextPageToken())) {
                logger.info("Pagination of backups failed. Initial list of backups was:");
                for (Backup backup : initialBackups) {
                    logger.info(backup.getId().toString());
                }
                logger.info("Current list of backups is:");
                ArrayList currentBackups = Lists.newArrayList((Iterable)dbAdminClient.listBackups(instanceId, new Options.ListOption[0]).iterateAll());
                for (Backup backup : currentBackups) {
                    logger.info(backup.getId().toString());
                }
            }
            Assert.assertFalse((boolean)Iterables.contains(seenPageTokens, (Object)page.getNextPageToken()));
            seenPageTokens.add(page.getNextPageToken());
            page = dbAdminClient.listBackups(instanceId, new Options.ListOption[]{Options.pageToken((String)page.getNextPageToken()), Options.pageSize((int)1)});
            Assert.assertEquals((long)1L, (long)Iterables.size((Iterable)page.getValues()));
            ++numBackups;
        }
        Assert.assertTrue((numBackups >= 1 ? 1 : 0) != 0);
    }

    private void testRestore(Backup backup, Timestamp versionTime, String expectedKey) throws InterruptedException, ExecutionException {
        String restoreOperationName;
        OperationFuture restoreOperation;
        String restoredDb = testHelper.getUniqueDatabaseId();
        int attempts = 0;
        while (true) {
            try {
                logger.info(String.format("Restoring backup %s to database %s", backup.getId().getBackup(), restoredDb));
                Restore restore = dbAdminClient.newRestoreBuilder(backup.getId(), DatabaseId.of((String)projectId, (String)instanceId, (String)restoredDb)).setEncryptionConfig((RestoreEncryptionConfig)EncryptionConfigs.customerManagedEncryption((String)expectedKey)).build();
                restoreOperation = dbAdminClient.restoreDatabase(restore);
                restoreOperationName = restoreOperation.getName();
            }
            catch (ExecutionException e) {
                if (e.getCause() instanceof FailedPreconditionException && e.getCause().getMessage().contains("Please retry the operation once the pending restores complete")) {
                    if (++attempts == 10) {
                        logger.info("Restore operation failed 10 times because of other pending restores. Skipping restore test.");
                        return;
                    }
                    logger.info(String.format("Restoring backup %s to database %s must wait because of other pending restore operation", backup.getId().getBackup(), restoredDb));
                    Thread.sleep(60000L);
                    continue;
                }
                throw e;
            }
            break;
        }
        databases.add(restoredDb);
        logger.info(String.format("Restore operation %s running", restoreOperationName));
        RestoreDatabaseMetadata metadata = (RestoreDatabaseMetadata)restoreOperation.getMetadata().get();
        Assert.assertEquals((Object)backup.getId().getName(), (Object)metadata.getBackupInfo().getBackup());
        Assert.assertEquals((Object)RestoreSourceType.BACKUP, (Object)metadata.getSourceType());
        Assert.assertEquals((Object)DatabaseId.of((InstanceId)testHelper.getInstanceId(), (String)restoredDb).getName(), (Object)metadata.getName());
        Assert.assertEquals((Object)versionTime, (Object)Timestamp.fromProto((com.google.protobuf.Timestamp)metadata.getBackupInfo().getVersionTime()));
        Database database = (Database)restoreOperation.get();
        Assert.assertEquals((Object)restoredDb, (Object)database.getId().getDatabase());
        Database reloadedDatabase = database.reload();
        Assert.assertNotNull((Object)reloadedDatabase.getProto());
        Assert.assertEquals((Object)versionTime, (Object)Timestamp.fromProto((com.google.protobuf.Timestamp)reloadedDatabase.getProto().getRestoreInfo().getBackupInfo().getVersionTime()));
        this.testDatabaseEncryption(reloadedDatabase, expectedKey);
        this.testDatabaseDialect(reloadedDatabase, Dialect.GOOGLE_STANDARD_SQL);
        logger.info(String.format("Restoring backup %s to existing database %s", backup.getId().getBackup(), restoredDb));
        ExecutionException executionException = (ExecutionException)Assert.assertThrows(ExecutionException.class, () -> backup.restore(DatabaseId.of((InstanceId)testHelper.getInstanceId(), (String)restoredDb)).get());
        Assert.assertEquals(SpannerException.class, executionException.getCause().getClass());
        SpannerException spannerException = (SpannerException)executionException.getCause();
        Assert.assertEquals((Object)ErrorCode.ALREADY_EXISTS, (Object)spannerException.getErrorCode());
    }

    private void verifyRestoreOperations(String backupOperationName, String restoreOperationName) {
        Assert.assertTrue((boolean)StreamSupport.stream(instance.listBackupOperations(new Options.ListOption[0]).iterateAll().spliterator(), false).anyMatch(input -> input.getName().equals(backupOperationName)));
        Assert.assertFalse((boolean)StreamSupport.stream(instance.listBackupOperations(new Options.ListOption[0]).iterateAll().spliterator(), false).anyMatch(input -> input.getName().equals(restoreOperationName)));
        Assert.assertFalse((boolean)StreamSupport.stream(instance.listDatabaseOperations(new Options.ListOption[0]).iterateAll().spliterator(), false).anyMatch(input -> input.getName().equals(backupOperationName)));
        Assert.assertTrue((boolean)StreamSupport.stream(instance.listDatabaseOperations(new Options.ListOption[0]).iterateAll().spliterator(), false).anyMatch(input -> input.getName().equals(restoreOperationName)));
    }

    static {
        databases = new ArrayList<String>();
        backups = new ArrayList<String>();
    }

    private static final class InjectErrorInterceptorProvider
    implements GrpcInterceptorProvider {
        final AtomicBoolean injectError = new AtomicBoolean(true);
        final AtomicInteger getOperationCount = new AtomicInteger();
        final AtomicInteger methodCount = new AtomicInteger();
        final String methodName;

        private InjectErrorInterceptorProvider(String methodName) {
            this.methodName = methodName;
        }

        public List<ClientInterceptor> getInterceptors() {
            ClientInterceptor interceptor = new ClientInterceptor(){

                public <ReqT, RespT> ClientCall<ReqT, RespT> interceptCall(MethodDescriptor<ReqT, RespT> method, CallOptions callOptions, Channel next) {
                    if (method.getFullMethodName().contains("GetOperation")) {
                        getOperationCount.incrementAndGet();
                    }
                    if (!method.getFullMethodName().contains(methodName)) {
                        return next.newCall(method, callOptions);
                    }
                    methodCount.incrementAndGet();
                    final AtomicBoolean errorInjected = new AtomicBoolean();
                    final ClientCall clientCall = next.newCall(method, callOptions);
                    return new ForwardingClientCall.SimpleForwardingClientCall<ReqT, RespT>(clientCall){

                        public void start(ClientCall.Listener<RespT> responseListener, Metadata headers) {
                            super.start((ClientCall.Listener)new ForwardingClientCallListener.SimpleForwardingClientCallListener<RespT>(responseListener){

                                public void onMessage(RespT message) {
                                    if (injectError.getAndSet(false)) {
                                        errorInjected.set(true);
                                        clientCall.cancel("Cancelling call for injected error", null);
                                    } else {
                                        super.onMessage(message);
                                    }
                                }

                                public void onClose(Status status, Metadata metadata) {
                                    if (errorInjected.get()) {
                                        status = Status.UNAVAILABLE.augmentDescription("INJECTED BY TEST");
                                    }
                                    super.onClose(status, metadata);
                                }
                            }, headers);
                        }
                    };
                }
            };
            return Collections.singletonList(interceptor);
        }
    }
}

