/*
 * Copyright 2015 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.cloud.storage;

import com.google.api.client.googleapis.json.GoogleJsonError;
import com.google.api.core.InternalApi;
import com.google.cloud.BaseServiceException;
import com.google.cloud.RetryHelper.RetryHelperException;
import com.google.cloud.http.BaseHttpServiceException;
import com.google.common.collect.ImmutableSet;
import java.io.IOException;
import java.util.Set;

/**
 * Storage service exception.
 *
 * @see <a href="https://cloud.google.com/storage/docs/json_api/v1/status-codes">Google Cloud
 *     Storage error codes</a>
 */
@InternalApi
public final class StorageException extends BaseHttpServiceException {
  private static final String INTERNAL_ERROR = "internalError";
  private static final String CONNECTION_CLOSED_PREMATURELY = "connectionClosedPrematurely";

  // see: https://cloud.google.com/storage/docs/resumable-uploads-xml#practices
  static final Set<Error> RETRYABLE_ERRORS =
      ImmutableSet.of(
          new Error(504, null),
          new Error(503, null),
          new Error(502, null),
          new Error(500, null),
          new Error(429, null),
          new Error(408, null),
          new Error(null, INTERNAL_ERROR),
          new Error(null, CONNECTION_CLOSED_PREMATURELY));

  private static final long serialVersionUID = -4168430271327813063L;

  public StorageException(int code, String message) {
    this(code, message, null);
  }

  public StorageException(int code, String message, Throwable cause) {
    super(code, message, null, true, RETRYABLE_ERRORS, cause);
  }

  public StorageException(int code, String message, String reason, Throwable cause) {
    super(code, message, reason, true, RETRYABLE_ERRORS, cause);
  }

  public StorageException(IOException exception) {
    super(exception, true, RETRYABLE_ERRORS);
  }

  public StorageException(GoogleJsonError error) {
    super(error, true, RETRYABLE_ERRORS);
  }

  /**
   * Translate RetryHelperException to the StorageException that caused the error. This method will
   * always throw an exception.
   *
   * @throws StorageException when {@code ex} was caused by a {@code StorageException}
   */
  public static StorageException translateAndThrow(RetryHelperException ex) {
    BaseServiceException.translate(ex);
    throw getStorageException(ex);
  }

  private static StorageException getStorageException(Throwable t) {
    return new StorageException(UNKNOWN_CODE, t.getMessage(), t.getCause());
  }

  /**
   * Attempt to find an Exception which is a {@link BaseServiceException} If neither {@code t} or
   * {@code t.getCause()} are a {@code BaseServiceException} a {@link StorageException} will be
   * created with an unknown status code.
   */
  static BaseServiceException coalesce(Throwable t) {
    if (t instanceof BaseServiceException) {
      return (BaseServiceException) t;
    }
    if (t.getCause() instanceof BaseServiceException) {
      return (BaseServiceException) t.getCause();
    }
    return getStorageException(t);
  }

  /**
   * Translate IOException to a StorageException representing the cause of the error. This method
   * defaults to idempotent always being {@code true}. Additionally, this method translates
   * transient issues Connection Closed Prematurely as a retryable error.
   *
   * @returns {@code StorageException}
   */
  public static StorageException translate(IOException exception) {
    String message = exception.getMessage();
    if (message != null
        && (message.contains("Connection closed prematurely")
            || message.contains("Premature EOF"))) {
      return new StorageException(0, message, CONNECTION_CLOSED_PREMATURELY, exception);
    } else {
      // default
      return new StorageException(exception);
    }
  }
}
