/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigquery.connector.common;

import com.google.cloud.RetryOption;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.BigQueryOptions;
import com.google.cloud.bigquery.Clustering;
import com.google.cloud.bigquery.Dataset;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.FieldValueList;
import com.google.cloud.bigquery.FormatOptions;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.bigquery.JobId;
import com.google.cloud.bigquery.JobInfo;
import com.google.cloud.bigquery.LoadJobConfiguration;
import com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.TableResult;
import com.google.cloud.bigquery.TimePartitioning;
import com.google.cloud.bigquery.connector.common.BigQueryConnectorException;
import com.google.cloud.bigquery.connector.common.BigQueryErrorCode;
import com.google.cloud.bigquery.connector.common.BigQueryUtil;
import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.threeten.bp.Duration;

public class BigQueryClient {
    private static final Logger log = LoggerFactory.getLogger(BigQueryClient.class);
    private final BigQuery bigQuery;
    private final Cache<String, TableInfo> destinationTableCache;
    private final Optional<String> materializationProject;
    private final Optional<String> materializationDataset;
    private final JobConfigurationFactory jobConfigurationFactory;

    public BigQueryClient(BigQuery bigQuery, Optional<String> materializationProject, Optional<String> materializationDataset, Cache<String, TableInfo> destinationTableCache, Map<String, String> labels, QueryJobConfiguration.Priority queryJobPriority) {
        this.bigQuery = bigQuery;
        this.materializationProject = materializationProject;
        this.materializationDataset = materializationDataset;
        this.destinationTableCache = destinationTableCache;
        this.jobConfigurationFactory = new JobConfigurationFactory(labels, queryJobPriority);
    }

    public static void waitForJob(Job job) {
        try {
            Job completedJob = job.waitFor(new RetryOption[]{RetryOption.initialRetryDelay((Duration)Duration.ofSeconds((long)1L)), RetryOption.totalTimeout((Duration)Duration.ofMinutes((long)3L))});
            if (completedJob == null && completedJob.getStatus().getError() != null) {
                throw new UncheckedIOException(new IOException(completedJob.getStatus().getError().toString()));
            }
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Could not copy table from temporary sink to destination table.", e);
        }
    }

    private static Optional<String> createWhereClause(String[] filters) {
        if (filters.length == 0) {
            return Optional.empty();
        }
        return Optional.of(Stream.of(filters).collect(Collectors.joining(") AND (", "(", ")")));
    }

    public TableInfo getTable(TableId tableId) {
        return this.bigQuery.getTable(tableId, new BigQuery.TableOption[0]);
    }

    public boolean tableExists(TableId tableId) {
        return this.getTable(tableId) != null;
    }

    public TableInfo createTable(TableId tableId, Schema schema) {
        TableInfo tableInfo = TableInfo.newBuilder((TableId)tableId, (TableDefinition)StandardTableDefinition.of((Schema)schema)).build();
        return this.bigQuery.create(tableInfo, new BigQuery.TableOption[0]);
    }

    public TableInfo createTempTable(TableId destinationTableId, Schema schema) {
        TableId tempTableId = this.createTempTableId(destinationTableId);
        TableInfo tableInfo = TableInfo.newBuilder((TableId)tempTableId, (TableDefinition)StandardTableDefinition.of((Schema)schema)).setExpirationTime(Long.valueOf(System.currentTimeMillis() + TimeUnit.DAYS.toMillis(1L))).build();
        return this.bigQuery.create(tableInfo, new BigQuery.TableOption[0]);
    }

    public TableId createTempTableId(TableId destinationTableId) {
        String tempProject = this.materializationProject.orElseGet(() -> ((TableId)destinationTableId).getProject());
        String tempDataset = this.materializationDataset.orElseGet(() -> ((TableId)destinationTableId).getDataset());
        String tableName = destinationTableId.getTable() + System.nanoTime();
        TableId tempTableId = tempProject == null ? TableId.of((String)tempDataset, (String)tableName) : TableId.of((String)tempProject, (String)tempDataset, (String)tableName);
        return tempTableId;
    }

    public boolean deleteTable(TableId tableId) {
        return this.bigQuery.delete(tableId);
    }

    public Job overwriteDestinationWithTemporary(TableId temporaryTableId, TableId destinationTableId) {
        String queryFormat = "MERGE `%s`\nUSING (SELECT * FROM `%s`)\nON FALSE\nWHEN NOT MATCHED THEN INSERT ROW\nWHEN NOT MATCHED BY SOURCE THEN DELETE";
        QueryJobConfiguration queryConfig = this.jobConfigurationFactory.createQueryJobConfigurationBuilder(this.sqlFromFormat(queryFormat, destinationTableId, temporaryTableId), Collections.emptyMap()).setUseLegacySql(Boolean.valueOf(false)).build();
        return this.create(JobInfo.newBuilder((JobConfiguration)queryConfig).build());
    }

    String sqlFromFormat(String queryFormat, TableId destinationTableId, TableId temporaryTableId) {
        String destinationTableName = BigQueryClient.fullTableName(destinationTableId);
        String temporaryTableName = BigQueryClient.fullTableName(temporaryTableId);
        return String.format(queryFormat, destinationTableName, temporaryTableName);
    }

    public String createTablePathForBigQueryStorage(TableId tableId) {
        Preconditions.checkNotNull((Object)tableId, (Object)"tableId cannot be null");
        String project = tableId.getProject() != null ? tableId.getProject() : this.getProjectId();
        return String.format("projects/%s/datasets/%s/tables/%s", project, tableId.getDataset(), tableId.getTable());
    }

    public TableInfo getReadTable(ReadTableOptions options) {
        Optional<String> query = options.query();
        if (query.isPresent()) {
            this.validateViewsEnabled(options);
            String sql = query.get();
            return this.materializeQueryToTable(sql, options.expirationTimeInMinutes());
        }
        TableInfo table = this.getTable(options.tableId());
        if (table == null) {
            return null;
        }
        TableDefinition tableDefinition = table.getDefinition();
        TableDefinition.Type tableType = tableDefinition.getType();
        if (TableDefinition.Type.TABLE == tableType || TableDefinition.Type.EXTERNAL == tableType || TableDefinition.Type.SNAPSHOT == tableType) {
            return table;
        }
        if (TableDefinition.Type.VIEW == tableType || TableDefinition.Type.MATERIALIZED_VIEW == tableType) {
            this.validateViewsEnabled(options);
            return table;
        }
        throw new BigQueryConnectorException(BigQueryErrorCode.UNSUPPORTED, String.format("Table type '%s' of table '%s.%s' is not supported", tableType, table.getTableId().getDataset(), table.getTableId().getTable()));
    }

    private void validateViewsEnabled(ReadTableOptions options) {
        if (!options.viewsEnabled()) {
            throw new BigQueryConnectorException(BigQueryErrorCode.UNSUPPORTED, String.format("Views are not enabled. You can enable views by setting '%s' to true. Notice additional cost may occur.", options.viewEnabledParamName()));
        }
    }

    DatasetId toDatasetId(TableId tableId) {
        return DatasetId.of((String)tableId.getProject(), (String)tableId.getDataset());
    }

    public String getProjectId() {
        return ((BigQueryOptions)this.bigQuery.getOptions()).getProjectId();
    }

    Iterable<Dataset> listDatasets(String projectId) {
        return this.bigQuery.listDatasets(projectId, new BigQuery.DatasetListOption[0]).iterateAll();
    }

    Iterable<Table> listTables(DatasetId datasetId, TableDefinition.Type ... types) {
        ImmutableSet allowedTypes = ImmutableSet.copyOf((Object[])types);
        Iterable allTables = this.bigQuery.listTables(datasetId, new BigQuery.TableListOption[0]).iterateAll();
        return (Iterable)StreamSupport.stream(allTables.spliterator(), false).filter(arg_0 -> BigQueryClient.lambda$listTables$0((Set)allowedTypes, arg_0)).collect(ImmutableList.toImmutableList());
    }

    TableId createDestinationTable(Optional<String> referenceProject, Optional<String> referenceDataset) {
        String project = this.materializationProject.orElse(referenceProject.orElse(null));
        String dataset = this.materializationDataset.orElse(referenceDataset.orElse(null));
        String name = String.format("_bqc_%s", UUID.randomUUID().toString().toLowerCase(Locale.ENGLISH).replace("-", ""));
        return project == null ? TableId.of((String)dataset, (String)name) : TableId.of((String)project, (String)dataset, (String)name);
    }

    public Table update(TableInfo table) {
        return this.bigQuery.update(table, new BigQuery.TableOption[0]);
    }

    public Job createAndWaitFor(JobConfiguration.Builder jobConfiguration) {
        return this.createAndWaitFor(jobConfiguration.build());
    }

    public Job createAndWaitFor(JobConfiguration jobConfiguration) {
        JobInfo jobInfo = JobInfo.of((JobConfiguration)jobConfiguration);
        Job job = this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]);
        log.info("Submitted job {}. jobId: {}", (Object)jobConfiguration, (Object)job.getJobId());
        try {
            return job.waitFor(new RetryOption[0]);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BigQueryException(0, String.format("Failed to run the job [%s]", job), (Throwable)e);
        }
    }

    Job create(JobInfo jobInfo) {
        return this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]);
    }

    public TableResult query(String sql) {
        try {
            return this.bigQuery.query(this.jobConfigurationFactory.createQueryJobConfigurationBuilder(sql, Collections.emptyMap()).build(), new BigQuery.JobOption[0]);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BigQueryException(0, String.format("Failed to run the query [%s]", sql), (Throwable)e);
        }
    }

    String createSql(TableId table, ImmutableList<String> requiredColumns, String[] filters) {
        String columns = requiredColumns.isEmpty() ? "*" : requiredColumns.stream().map(column -> String.format("`%s`", column)).collect(Collectors.joining(","));
        return this.createSql(table, columns, filters);
    }

    String createSql(TableId table, String formattedQuery, String[] filters) {
        String tableName = BigQueryClient.fullTableName(table);
        String whereClause = BigQueryClient.createWhereClause(filters).map(clause -> "WHERE " + clause).orElse("");
        return String.format("SELECT %s FROM `%s` %s", formattedQuery, tableName, whereClause);
    }

    public static String fullTableName(TableId tableId) {
        if (tableId.getProject() == null) {
            return String.format("%s.%s", tableId.getDataset(), tableId.getTable());
        }
        return String.format("%s.%s.%s", tableId.getProject(), tableId.getDataset(), tableId.getTable());
    }

    public long calculateTableSize(TableId tableId, Optional<String> filter) {
        return this.calculateTableSize(this.getTable(tableId), filter);
    }

    public long calculateTableSize(TableInfo tableInfo, Optional<String> filter) {
        TableDefinition.Type type = tableInfo.getDefinition().getType();
        if (type == TableDefinition.Type.TABLE && !filter.isPresent()) {
            return tableInfo.getNumRows().longValue();
        }
        if (type == TableDefinition.Type.EXTERNAL && !filter.isPresent()) {
            String table = BigQueryClient.fullTableName(tableInfo.getTableId());
            return this.getNumberOfRows(String.format("SELECT COUNT(*) from `%s`", table));
        }
        if (type == TableDefinition.Type.VIEW || type == TableDefinition.Type.MATERIALIZED_VIEW || (type == TableDefinition.Type.TABLE || type == TableDefinition.Type.EXTERNAL) && filter.isPresent()) {
            String table = BigQueryClient.fullTableName(tableInfo.getTableId());
            String whereClause = filter.map(f -> "WHERE " + f).orElse("");
            return this.getNumberOfRows(String.format("SELECT COUNT(*) from `%s` %s", table, whereClause));
        }
        throw new IllegalArgumentException(String.format("Unsupported table type %s for table %s", type, BigQueryClient.fullTableName(tableInfo.getTableId())));
    }

    private long getNumberOfRows(String sql) {
        TableResult result = this.query(sql);
        long numberOfRows = ((FieldValueList)result.iterateAll().iterator().next()).get(0).getLongValue();
        return numberOfRows;
    }

    public TableInfo materializeQueryToTable(String querySql, int expirationTimeInMinutes) {
        TableId tableId = this.createDestinationTable(Optional.empty(), Optional.empty());
        return this.materializeTable(querySql, tableId, expirationTimeInMinutes);
    }

    public TableInfo materializeQueryToTable(String querySql, int expirationTimeInMinutes, Map<String, String> additionalQueryJobLabels) {
        TableId destinationTableId = this.createDestinationTable(Optional.empty(), Optional.empty());
        DestinationTableBuilder tableBuilder = new DestinationTableBuilder(this, querySql, destinationTableId, expirationTimeInMinutes, this.jobConfigurationFactory, additionalQueryJobLabels);
        return this.materializeTable(querySql, tableBuilder);
    }

    public TableInfo materializeViewToTable(String querySql, TableId viewId, int expirationTimeInMinutes) {
        TableId tableId = this.createDestinationTable(Optional.ofNullable(viewId.getProject()), Optional.ofNullable(viewId.getDataset()));
        return this.materializeTable(querySql, tableId, expirationTimeInMinutes);
    }

    private TableInfo materializeTable(String querySql, TableId destinationTableId, int expirationTimeInMinutes) {
        try {
            return (TableInfo)this.destinationTableCache.get((Object)querySql, (Callable)new DestinationTableBuilder(this, querySql, destinationTableId, expirationTimeInMinutes, this.jobConfigurationFactory, Collections.emptyMap()));
        }
        catch (Exception e) {
            throw new BigQueryConnectorException(BigQueryErrorCode.BIGQUERY_VIEW_DESTINATION_TABLE_CREATION_FAILED, String.format("Error creating destination table using the following query: [%s]", querySql), e);
        }
    }

    private TableInfo materializeTable(String querySql, DestinationTableBuilder destinationTableBuilder) {
        try {
            return (TableInfo)this.destinationTableCache.get((Object)querySql, (Callable)destinationTableBuilder);
        }
        catch (Exception e) {
            throw new BigQueryConnectorException(BigQueryErrorCode.BIGQUERY_VIEW_DESTINATION_TABLE_CREATION_FAILED, String.format("Error creating destination table using the following query: [%s]", querySql), e);
        }
    }

    public void loadDataIntoTable(LoadDataOptions options, List<String> sourceUris, FormatOptions formatOptions, JobInfo.WriteDisposition writeDisposition, Optional<Schema> schema) {
        LoadJobConfiguration.Builder jobConfiguration = this.jobConfigurationFactory.createLoadJobConfigurationBuilder(options, sourceUris, formatOptions).setCreateDisposition(JobInfo.CreateDisposition.CREATE_IF_NEEDED).setWriteDisposition(writeDisposition);
        if (schema.isPresent()) {
            jobConfiguration.setSchema(schema.get());
        } else {
            jobConfiguration.setAutodetect(Boolean.valueOf(true));
        }
        options.getCreateDisposition().ifPresent(arg_0 -> ((LoadJobConfiguration.Builder)jobConfiguration).setCreateDisposition(arg_0));
        if (options.getPartitionField().isPresent() || options.getPartitionType().isPresent()) {
            TimePartitioning.Builder timePartitionBuilder = TimePartitioning.newBuilder((TimePartitioning.Type)options.getPartitionTypeOrDefault());
            options.getPartitionExpirationMs().ifPresent(arg_0 -> ((TimePartitioning.Builder)timePartitionBuilder).setExpirationMs(arg_0));
            options.getPartitionRequireFilter().ifPresent(arg_0 -> ((TimePartitioning.Builder)timePartitionBuilder).setRequirePartitionFilter(arg_0));
            options.getPartitionField().ifPresent(arg_0 -> ((TimePartitioning.Builder)timePartitionBuilder).setField(arg_0));
            jobConfiguration.setTimePartitioning(timePartitionBuilder.build());
        }
        options.getClusteredFields().ifPresent(clusteredFields -> {
            Clustering clustering = Clustering.newBuilder().setFields((List)clusteredFields).build();
            jobConfiguration.setClustering(clustering);
        });
        if (options.isUseAvroLogicalTypes()) {
            jobConfiguration.setUseAvroLogicalTypes(Boolean.valueOf(true));
        }
        if (!options.getDecimalTargetTypes().isEmpty()) {
            jobConfiguration.setDecimalTargetTypes(options.getDecimalTargetTypes());
        }
        if (!options.getLoadSchemaUpdateOptions().isEmpty()) {
            jobConfiguration.setSchemaUpdateOptions(options.getLoadSchemaUpdateOptions());
        }
        Job finishedJob = null;
        try {
            finishedJob = this.createAndWaitFor((JobConfiguration.Builder)jobConfiguration);
            if (finishedJob.getStatus().getError() != null) {
                throw new BigQueryException(0, String.format("Failed to load to %s in job %s. BigQuery error was '%s'", BigQueryUtil.friendlyTableName(options.getTableId()), finishedJob.getJobId(), finishedJob.getStatus().getError().getMessage()), finishedJob.getStatus().getError());
            }
            log.info("Done loading to {}. jobId: {}", (Object)BigQueryUtil.friendlyTableName(options.getTableId()), (Object)finishedJob.getJobId());
        }
        catch (Exception e) {
            if (finishedJob == null) {
                log.error("Unable to create the job to load to {}", (Object)BigQueryUtil.friendlyTableName(options.getTableId()));
                throw e;
            }
            TimePartitioning.Type partitionType = options.getPartitionTypeOrDefault();
            if (e.getMessage().equals(String.format("Cannot output %s partitioned data in LegacySQL", partitionType)) && formatOptions.equals((Object)FormatOptions.parquet())) {
                throw new BigQueryException(0, String.format("%s time partitioning is not available for load jobs from PARQUET in this project yet. Please replace the intermediate format to AVRO or contact your account manager to enable this.", partitionType), (Throwable)e);
            }
            JobId jobId = finishedJob.getJobId();
            log.warn(String.format("Failed to load the data into BigQuery, JobId for debug purposes is [%s:%s.%s]", jobId.getProject(), jobId.getLocation(), jobId.getJob()));
            throw new BigQueryException(0, "Problem loading data into BigQuery", (Throwable)e);
        }
    }

    public void createTableIfNeeded(TableId tableId, Schema bigQuerySchema) {
        if (!this.tableExists(tableId)) {
            this.createTable(tableId, bigQuerySchema);
        }
    }

    private static /* synthetic */ boolean lambda$listTables$0(Set allowedTypes, Table table) {
        return allowedTypes.contains(table.getDefinition().getType());
    }

    static class JobConfigurationFactory {
        private final ImmutableMap<String, String> labels;
        private final QueryJobConfiguration.Priority queryJobPriority;

        public JobConfigurationFactory(Map<String, String> labels, QueryJobConfiguration.Priority queryJobPriority) {
            this.labels = ImmutableMap.copyOf(labels);
            this.queryJobPriority = queryJobPriority;
        }

        QueryJobConfiguration.Builder createQueryJobConfigurationBuilder(String querySql, Map<String, String> additionalQueryJobLabels) {
            QueryJobConfiguration.Builder builder = QueryJobConfiguration.newBuilder((String)querySql).setPriority(this.queryJobPriority);
            HashMap<String, String> allLabels = new HashMap<String, String>(additionalQueryJobLabels);
            if (this.labels != null && !this.labels.isEmpty()) {
                allLabels.putAll((Map<String, String>)this.labels);
            }
            builder.setLabels(allLabels);
            return builder;
        }

        LoadJobConfiguration.Builder createLoadJobConfigurationBuilder(LoadDataOptions options, List<String> sourceUris, FormatOptions formatOptions) {
            LoadJobConfiguration.Builder builder = LoadJobConfiguration.newBuilder((TableId)options.getTableId(), sourceUris, (FormatOptions)formatOptions);
            if (this.labels != null && !this.labels.isEmpty()) {
                builder.setLabels(this.labels);
            }
            return builder;
        }
    }

    static class DestinationTableBuilder
    implements Callable<TableInfo> {
        final BigQueryClient bigQueryClient;
        final String querySql;
        final TableId destinationTable;
        final int expirationTimeInMinutes;
        final JobConfigurationFactory jobConfigurationFactory;
        final Map<String, String> additionalQueryJobLabels;

        DestinationTableBuilder(BigQueryClient bigQueryClient, String querySql, TableId destinationTable, int expirationTimeInMinutes, JobConfigurationFactory jobConfigurationFactory, Map<String, String> additionalQueryJobLabels) {
            this.bigQueryClient = bigQueryClient;
            this.querySql = querySql;
            this.destinationTable = destinationTable;
            this.expirationTimeInMinutes = expirationTimeInMinutes;
            this.jobConfigurationFactory = jobConfigurationFactory;
            this.additionalQueryJobLabels = additionalQueryJobLabels;
        }

        @Override
        public TableInfo call() {
            return this.createTableFromQuery();
        }

        TableInfo createTableFromQuery() {
            log.debug("destinationTable is %s", (Object)this.destinationTable);
            JobInfo jobInfo = JobInfo.of((JobConfiguration)this.jobConfigurationFactory.createQueryJobConfigurationBuilder(this.querySql, this.additionalQueryJobLabels).setDestinationTable(this.destinationTable).build());
            log.debug("running query %s", (Object)jobInfo);
            Job job = this.waitForJob(this.bigQueryClient.create(jobInfo));
            log.debug("job has finished. %s", (Object)job);
            if (job.getStatus().getError() != null) {
                throw BigQueryUtil.convertToBigQueryException(job.getStatus().getError());
            }
            TableInfo createdTable = this.bigQueryClient.getTable(this.destinationTable);
            long expirationTime = createdTable.getCreationTime() + TimeUnit.MINUTES.toMillis(this.expirationTimeInMinutes);
            Table updatedTable = this.bigQueryClient.update(createdTable.toBuilder().setExpirationTime(Long.valueOf(expirationTime)).build());
            return updatedTable;
        }

        Job waitForJob(Job job) {
            try {
                return job.waitFor(new RetryOption[0]);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new BigQueryException(0, String.format("Job %s has been interrupted", job.getJobId()), (Throwable)e);
            }
        }
    }

    public static interface LoadDataOptions {
        public TableId getTableId();

        public Optional<JobInfo.CreateDisposition> getCreateDisposition();

        public Optional<String> getPartitionField();

        public Optional<TimePartitioning.Type> getPartitionType();

        public TimePartitioning.Type getPartitionTypeOrDefault();

        public OptionalLong getPartitionExpirationMs();

        public Optional<Boolean> getPartitionRequireFilter();

        public Optional<ImmutableList<String>> getClusteredFields();

        public boolean isUseAvroLogicalTypes();

        public List<String> getDecimalTargetTypes();

        public List<JobInfo.SchemaUpdateOption> getLoadSchemaUpdateOptions();

        public boolean getEnableModeCheckForSchemaFields();
    }

    public static interface ReadTableOptions {
        public TableId tableId();

        public Optional<String> query();

        public boolean viewsEnabled();

        public String viewEnabledParamName();

        public int expirationTimeInMinutes();
    }
}

