/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigquery.connector.common;

import com.google.cloud.bigquery.connector.common.BigQueryClient;
import com.google.cloud.bigquery.connector.common.BigQueryClientFactory;
import com.google.cloud.bigquery.connector.common.BigQueryConnectorException;
import com.google.cloud.bigquery.connector.common.BigQueryErrorCode;
import com.google.cloud.bigquery.connector.common.ReadSessionCreatorConfig;
import com.google.cloud.bigquery.connector.common.ReadSessionResponse;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableId;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableInfo;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.storage.v1.ArrowSerializationOptions;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.storage.v1.BigQueryReadClient;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.storage.v1.CreateReadSessionRequest;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.storage.v1.ReadSession;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.cache.Cache;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.cache.CacheBuilder;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.collect.ImmutableList;
import com.google.cloud.spark.bigquery.repackaged.com.google.gson.Gson;
import com.google.cloud.spark.bigquery.repackaged.com.google.gson.JsonObject;
import com.google.cloud.spark.bigquery.repackaged.com.google.protobuf.InvalidProtocolBufferException;
import com.google.cloud.spark.bigquery.repackaged.com.google.protobuf.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReadSessionCreator {
    public static final int DEFAULT_MAX_PARALLELISM = 20000;
    public static final int MINIMAL_PARALLELISM = 1;
    public static final int DEFAULT_MIN_PARALLELISM_FACTOR = 3;
    private static final Logger log = LoggerFactory.getLogger(ReadSessionCreator.class);
    private static boolean initialized = false;
    private static Cache<CreateReadSessionRequest, ReadSession> READ_SESSION_CACHE;
    private final ReadSessionCreatorConfig config;
    private final BigQueryClient bigQueryClient;
    private final BigQueryClientFactory bigQueryReadClientFactory;

    private static synchronized void initializeCache(long readSessionCacheDurationMins) {
        if (!initialized) {
            READ_SESSION_CACHE = CacheBuilder.newBuilder().expireAfterWrite(readSessionCacheDurationMins, TimeUnit.MINUTES).maximumSize(1000L).build();
            initialized = true;
        }
    }

    public ReadSessionCreator(ReadSessionCreatorConfig config, BigQueryClient bigQueryClient, BigQueryClientFactory bigQueryReadClientFactory) {
        this.config = config;
        this.bigQueryClient = bigQueryClient;
        this.bigQueryReadClientFactory = bigQueryReadClientFactory;
        ReadSessionCreator.initializeCache(config.getReadSessionCacheDurationMins());
    }

    public ReadSessionResponse create(TableId table, ImmutableList<String> selectedFields, Optional<String> filter) {
        Instant sessionPrepStartTime = Instant.now();
        TableInfo tableDetails = this.bigQueryClient.getTable(table);
        TableInfo actualTable = this.getActualTable(tableDetails, selectedFields, filter);
        BigQueryReadClient bigQueryReadClient = this.bigQueryReadClientFactory.getBigQueryReadClient();
        log.info("|creation a read session for table {}, parameters: |selectedFields=[{}],|filter=[{}]|snapshotTimeMillis[{}]", new Object[]{actualTable.getFriendlyName(), String.join((CharSequence)",", selectedFields), filter.orElse("None"), this.config.getSnapshotTimeMillis().isPresent() ? String.valueOf(this.config.getSnapshotTimeMillis().getAsLong()) : "None"});
        String tablePath = ReadSessionCreator.toTablePath(actualTable.getTableId());
        CreateReadSessionRequest request = this.config.getRequestEncodedBase().map(value -> {
            try {
                return CreateReadSessionRequest.parseFrom(Base64.getDecoder().decode((String)value));
            }
            catch (InvalidProtocolBufferException e) {
                throw new RuntimeException("Couldn't decode:" + value, e);
            }
        }).orElse(CreateReadSessionRequest.newBuilder().build());
        ReadSession.Builder requestedSession = request.getReadSession().toBuilder();
        this.config.getTraceId().ifPresent(traceId -> requestedSession.setTraceId((String)traceId));
        ReadSession.TableReadOptions.Builder readOptions = requestedSession.getReadOptionsBuilder();
        if (!this.isInputTableAView(tableDetails)) {
            filter.ifPresent(readOptions::setRowRestriction);
        }
        readOptions.addAllSelectedFields(selectedFields);
        readOptions.setArrowSerializationOptions(ArrowSerializationOptions.newBuilder().setBufferCompression(this.config.getArrowCompressionCodec()).build());
        readOptions.setResponseCompressionCodec(this.config.getResponseCompressionCodec());
        int preferredMinStreamCount = this.config.getPreferredMinParallelism().orElseGet(() -> {
            int defaultPreferredMinStreamCount = Math.max(1, 3 * this.config.getDefaultParallelism());
            log.debug("using default preferred min parallelism [{}]", (Object)defaultPreferredMinStreamCount);
            return defaultPreferredMinStreamCount;
        });
        int maxStreamCount = this.config.getMaxParallelism().orElseGet(() -> {
            int defaultMaxStreamCount = Math.max(20000, preferredMinStreamCount);
            log.debug("using default max parallelism [{}]", (Object)defaultMaxStreamCount);
            return defaultMaxStreamCount;
        });
        int minStreamCount = preferredMinStreamCount;
        if (minStreamCount > maxStreamCount) {
            minStreamCount = maxStreamCount;
            log.warn("preferred min parallelism is larger than the max parallelism, therefore setting it to max parallelism [{}]", (Object)minStreamCount);
        }
        Instant sessionPrepEndTime = Instant.now();
        ReadSession.TableModifiers.Builder modifiers = ReadSession.TableModifiers.newBuilder();
        if (!this.isInputTableAView(tableDetails)) {
            this.config.getSnapshotTimeMillis().ifPresent(millis -> {
                Instant snapshotTime = Instant.ofEpochMilli(millis);
                modifiers.setSnapshotTime(Timestamp.newBuilder().setSeconds(snapshotTime.getEpochSecond()).setNanos(snapshotTime.getNano()).build());
            });
        }
        CreateReadSessionRequest createReadSessionRequest = request.newBuilder().setParent("projects/" + this.bigQueryClient.getProjectId()).setReadSession(requestedSession.setDataFormat(this.config.getReadDataFormat()).setReadOptions(readOptions).setTableModifiers(modifiers).setTable(tablePath).build()).setMaxStreamCount(maxStreamCount).setPreferredMinStreamCount(minStreamCount).build();
        if (this.config.isReadSessionCachingEnabled() && this.getReadSessionCache().asMap().containsKey(createReadSessionRequest)) {
            ReadSession readSession = (ReadSession)this.getReadSessionCache().asMap().get(createReadSessionRequest);
            log.info("Reusing read session: {}, for table: {}", (Object)readSession.getName(), (Object)table);
            return new ReadSessionResponse(readSession, actualTable);
        }
        ReadSession readSession = bigQueryReadClient.createReadSession(createReadSessionRequest);
        if (readSession != null) {
            Instant sessionCreationEndTime = Instant.now();
            if (this.config.isReadSessionCachingEnabled()) {
                this.getReadSessionCache().put(createReadSessionRequest, readSession);
            }
            JsonObject jsonObject = new JsonObject();
            jsonObject.addProperty("readSessionName", readSession.getName());
            jsonObject.addProperty("readSessionCreationStartTime", sessionPrepStartTime.toString());
            jsonObject.addProperty("readSessionCreationEndTime", sessionCreationEndTime.toString());
            jsonObject.addProperty("readSessionPrepDuration", Duration.between(sessionPrepStartTime, sessionPrepEndTime).toMillis());
            jsonObject.addProperty("readSessionCreationDuration", Duration.between(sessionPrepEndTime, sessionCreationEndTime).toMillis());
            jsonObject.addProperty("readSessionDuration", Duration.between(sessionPrepStartTime, sessionCreationEndTime).toMillis());
            log.info("Read session:{}", (Object)new Gson().toJson(jsonObject));
            log.info("Received {} partitions from the BigQuery Storage API for session {}. Notice that the number of streams in actual may be lower than the requested number, depending on the amount parallelism that is reasonable for the table and the maximum amount of parallelism allowed by the system.", (Object)readSession.getStreamsCount(), (Object)readSession.getName());
        }
        return new ReadSessionResponse(readSession, actualTable);
    }

    static String toTablePath(TableId tableId) {
        return String.format("projects/%s/datasets/%s/tables/%s", tableId.getProject(), tableId.getDataset(), tableId.getTable());
    }

    public TableInfo getActualTable(TableInfo table, ImmutableList<String> requiredColumns, Optional<String> filter) {
        String[] filters = (String[])filter.map(Stream::of).orElseGet(Stream::empty).toArray(String[]::new);
        return this.getActualTable(table, requiredColumns, filters);
    }

    TableInfo getActualTable(TableInfo table, ImmutableList<String> requiredColumns, String[] filters) {
        Object tableDefinition = table.getDefinition();
        TableDefinition.Type tableType = ((TableDefinition)tableDefinition).getType();
        if (TableDefinition.Type.TABLE == tableType || TableDefinition.Type.EXTERNAL == tableType || TableDefinition.Type.SNAPSHOT == tableType) {
            return table;
        }
        if (this.isInputTableAView(table)) {
            String querySql = this.bigQueryClient.createSql(table.getTableId(), requiredColumns, filters, this.config.getSnapshotTimeMillis());
            log.debug("querySql is {}", (Object)querySql);
            return this.bigQueryClient.materializeViewToTable(querySql, table.getTableId(), this.config.getMaterializationExpirationTimeInMinutes());
        }
        throw new BigQueryConnectorException(BigQueryErrorCode.UNSUPPORTED, String.format("Table type '%s' of table '%s.%s' is not supported", tableType, table.getTableId().getDataset(), table.getTableId().getTable()));
    }

    public boolean isInputTableAView(TableInfo table) {
        Object tableDefinition = table.getDefinition();
        TableDefinition.Type tableType = ((TableDefinition)tableDefinition).getType();
        if (TableDefinition.Type.VIEW == tableType || TableDefinition.Type.MATERIALIZED_VIEW == tableType) {
            if (!this.config.isViewsEnabled()) {
                throw new BigQueryConnectorException(BigQueryErrorCode.UNSUPPORTED, String.format("Views are not enabled. You can enable views by setting '%s' to true. Notice additional cost may occur.", this.config.getViewEnabledParamName()));
            }
            return true;
        }
        return false;
    }

    Cache<CreateReadSessionRequest, ReadSession> getReadSessionCache() {
        return READ_SESSION_CACHE;
    }
}

