/*
 * Copyright 2017-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.cloud.spring.data.firestore.repository.config;

import com.google.cloud.spring.data.firestore.repository.support.FirestoreRepositoryFactoryBean;
import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Import;
import org.springframework.data.repository.config.DefaultRepositoryBaseClass;

/**
 * Annotation that enables Firestore's Query Method functionality.
 *
 * @since 1.1
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
@Inherited
@Documented
@Import(FirestoreRepositoriesRegistrar.class)
public @interface EnableReactiveFirestoreRepositories {

  /**
   * Alias for the {@link #basePackages()} attribute. Allows for more concise annotation
   * declarations e.g.: {@code @EnableReactiveFirestoreRepositories("org.my.pkg")} instead of
   * {@code @EnableReactiveFirestoreRepositories(basePackages="org.my.pkg")}.
   *
   * @return an empty array
   */
  String[] value() default {};

  /**
   * Specifies which types are eligible for component scanning. Further narrows the set of candidate
   * components from everything in {@link #basePackages()} to everything in the base packages that
   * matches the given filter or filters.
   *
   * @return an empty array.
   */
  ComponentScan.Filter[] includeFilters() default {};

  /**
   * Specifies which types are not eligible for component scanning.
   *
   * @return an empty array
   */
  ComponentScan.Filter[] excludeFilters() default {};

  /**
   * Base packages to scan for annotated components. {@link #value()} is an alias for (and mutually
   * exclusive with) this attribute. Use {@link #basePackageClasses()} for a type-safe alternative
   * to String-based package names.
   *
   * @return an empty array
   */
  String[] basePackages() default {};

  /**
   * Type-safe alternative to {@link #basePackages()} for specifying the packages to scan for
   * annotated components. The package of each class specified will be scanned. Consider creating a
   * special no-op marker class or interface in each package that serves no purpose other than being
   * referenced by this attribute.
   *
   * @return an empty array
   */
  Class[] basePackageClasses() default {};

  /**
   * Configure the repository base class to be used to create repository proxies for this particular
   * configuration.
   *
   * @return the base repository class
   */
  Class repositoryBaseClass() default DefaultRepositoryBaseClass.class;

  /**
   * Configures whether nested repository-interfaces (e.g. defined as inner classes) should be
   * discovered by the repositories infrastructure.
   *
   * @return false
   */
  boolean considerNestedRepositories() default false;

  /**
   * Returns the {@link org.springframework.beans.factory.FactoryBean} class to be used for each
   * repository instance. Defaults to {@link FirestoreRepositoryFactoryBean}.
   *
   * @return the factory bean class used to create factories
   */
  Class repositoryFactoryBeanClass() default FirestoreRepositoryFactoryBean.class;

  /**
   * Unused. Firestore does not support named queries.
   *
   * @return Unused. Forestore does not support named queries.
   */
  String namedQueriesLocation() default "";

  /**
   * Returns the postfix to be used when looking up custom repository implementations. Defaults to
   * {@literal Impl}. So for a repository named {@code PersonRepository} the corresponding
   * implementation class will be looked up scanning for {@code PersonRepositoryImpl}.
   *
   * @return the default suffix that will cause classes to be assumed to be implementations
   */
  String repositoryImplementationPostfix() default "";

  /**
   * Configures the name of the Firestore template bean to be used by default with the repositories
   * detected.
   *
   * @return the name of the Cloud Firestore template class
   */
  String firestoreTemplateRef() default "firestoreTemplate";

  /**
   * Configures the name of the Firestore mapping context bean to be used by default with the
   * repositories detected.
   *
   * @return the name of the Cloud Firestore mapping context class
   */
  String firestoreMappingContextRef() default "firestoreMappingContext";
}
