/*
 * Copyright 2000-2012 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ksp.com.intellij.core;

import ksp.com.intellij.openapi.diagnostic.Logger;
import ksp.com.intellij.openapi.roots.PackageIndex;
import ksp.com.intellij.openapi.vfs.VfsUtilCore;
import ksp.com.intellij.openapi.vfs.VirtualFile;
import ksp.com.intellij.util.CollectionQuery;
import ksp.com.intellij.util.Query;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

public class CorePackageIndex extends PackageIndex {
  private static final Logger LOG = Logger.getInstance(CorePackageIndex.class);

  private final List<VirtualFile> myClasspath = new ArrayList<>();

  public CorePackageIndex() {
  }

  private List<VirtualFile> roots() {
    return myClasspath;
  }

  private List<VirtualFile> findDirectoriesByPackageName(String packageName) {
    List<VirtualFile> result = new ArrayList<>();
    String dirName = packageName.replace(".", "/");
    for (VirtualFile root : roots()) {
      VirtualFile classDir = root.findFileByRelativePath(dirName);
      if (classDir != null) {
        result.add(classDir);
      }
    }
    return result;
  }

  @Override
  public VirtualFile @NotNull [] getDirectoriesByPackageName(@NotNull String packageName, boolean includeLibrarySources) {
    return getDirsByPackageName(packageName, includeLibrarySources).toArray(VirtualFile.EMPTY_ARRAY);
  }

  @NotNull
  @Override
  public Query<VirtualFile> getDirsByPackageName(@NotNull String packageName, boolean includeLibrarySources) {
    return new CollectionQuery<>(findDirectoriesByPackageName(packageName));
  }

  public void addToClasspath(VirtualFile root) {
    myClasspath.add(root);
  }

  @Override
  public @Nullable String getPackageNameByDirectory(@NotNull VirtualFile dir) {
    for (VirtualFile root : roots()) {
      if (VfsUtilCore.isAncestor(root, dir, false)) {
        return VfsUtilCore.getRelativePath(dir, root, '.');
      }
    }
    return null;
  }
}
