// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.

package ksp.com.intellij.extapi.psi;

import ksp.com.intellij.lang.ASTNode;
import ksp.com.intellij.lang.Language;
import ksp.com.intellij.openapi.application.ApplicationManager;
import ksp.com.intellij.openapi.diagnostic.Attachment;
import ksp.com.intellij.openapi.diagnostic.RuntimeExceptionWithAttachments;
import ksp.com.intellij.openapi.progress.ProgressIndicatorProvider;
import ksp.com.intellij.openapi.progress.ProgressManager;
import ksp.com.intellij.openapi.project.Project;
import ksp.com.intellij.openapi.project.ProjectCoreUtil;
import ksp.com.intellij.openapi.util.Key;
import ksp.com.intellij.openapi.util.RecursionManager;
import ksp.com.intellij.openapi.vfs.VirtualFile;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.psi.PsiInvalidElementAccessException;
import ksp.com.intellij.psi.impl.DebugUtil;
import ksp.com.intellij.psi.impl.PsiManagerEx;
import ksp.com.intellij.psi.impl.source.PsiFileImpl;
import ksp.com.intellij.psi.impl.source.SourceTreeToPsiMap;
import ksp.com.intellij.psi.impl.source.SubstrateRef;
import ksp.com.intellij.psi.impl.source.tree.CompositeElement;
import ksp.com.intellij.psi.impl.source.tree.FileElement;
import ksp.com.intellij.psi.impl.source.tree.RecursiveTreeElementWalkingVisitor;
import ksp.com.intellij.psi.impl.source.tree.SharedImplUtil;
import ksp.com.intellij.psi.stubs.*;
import ksp.com.intellij.psi.tree.IElementType;
import ksp.com.intellij.psi.tree.TokenSet;
import ksp.com.intellij.psi.util.PsiTreeUtil;
import ksp.com.intellij.util.ArrayFactory;
import ksp.com.intellij.util.ArrayUtil;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * A base class for PSI elements that support both stub and AST substrates. The purpose of stubs is to hold the most important information
 * (like element names, access modifiers, function parameters etc), save it in the index and use it during code analysis instead of parsing
 * the AST, which can be quite expensive. Ideally, only the files loaded in the editor should ever be parsed, all other information that's
 * needed e.g. for resolving references in those files, should be taken from stub-based PSI.<p/>
 *
 * During indexing, this element is created from text (using {@link #StubBasedPsiElementBase(ASTNode)}),
 * then a stub with relevant information is built from it
 * ({@link IStubElementType#createStub(PsiElement, StubElement)}), and this stub is saved in the
 * index. Then a StubIndex query returns an instance of this element based on a stub
 * (created with {@link #StubBasedPsiElementBase(StubElement, IStubElementType)} constructor. To the clients, such element looks exactly
 * like the one created from AST, all the methods should work the same way.<p/>
 *
 * Code analysis clients should be careful not to invoke methods on this class that can't be implemented using only the information from
 * stubs. Such case is supported: {@link #getNode()} will switch the implementation from stub to AST substrate and get this information, but
 * this is slow performance-wise. So stubs should be designed so that they hold all the information that's relevant for
 * reference resolution and other code analysis.<p/>
 *
 * The subclasses should be careful not to switch to AST prematurely. For example, {@link #getParentByStub()} should be used as much
 * as possible in overridden {@link #getParent()}, and getStubOrPsiChildren methods should be preferred over {@link #getChildren()}.<p/>
 *
 * After switching to AST, {@link #getStub()} will return null, but {@link #getGreenStub()} can still be used to retrieve stub objects if they're needed.
 * The AST itself is not held on a strong reference and can be garbage-collected. This makes it possible to hold many stub-based PSI elements
 * in the memory at once, but results in occasionally expensive {@link #getNode()} calls that have to load and parse the AST anew.
 *
 * @see IStubElementType
 * @see com.intellij.psi.impl.source.PsiFileWithStubSupport
 */
public class StubBasedPsiElementBase<T extends StubElement> extends ASTDelegatePsiElement {
  public static final Key<String> CREATION_TRACE = Key.create("CREATION_TRACE");
  public static final boolean ourTraceStubAstBinding = "true".equals(System.getProperty("trace.stub.ast.binding", "false"));
  private volatile SubstrateRef mySubstrateRef;
  private final IElementType myElementType;

  public StubBasedPsiElementBase(@NotNull T stub, @NotNull IStubElementType<?,?> nodeType) {
    mySubstrateRef = new SubstrateRef.StubRef(stub);
    myElementType = nodeType;
  }

  public StubBasedPsiElementBase(@NotNull ASTNode node) {
    mySubstrateRef = SubstrateRef.createAstStrongRef(node);
    myElementType = node.getElementType();
  }

  /**
   * This constructor is created to allow inheriting from this class in JVM languages which doesn't support multiple constructors (e.g. Scala).
   * If your language does support multiple constructors use {@link #StubBasedPsiElementBase(StubElement, IStubElementType)} and
   * {@link #StubBasedPsiElementBase(ASTNode)} instead.
   */
  public StubBasedPsiElementBase(T stub, IElementType nodeType, ASTNode node) {
    if (stub != null) {
      if (nodeType == null) throw new IllegalArgumentException("null cannot be passed to 'nodeType' when 'stub' is non-null");
      if (node != null) throw new IllegalArgumentException("null must be passed to 'node' parameter when 'stub' is non-null");
      mySubstrateRef = new SubstrateRef.StubRef(stub);
      myElementType = nodeType;
    }
    else {
      if (node == null) throw new IllegalArgumentException("'stub' and 'node' parameters cannot be null both");
      if (nodeType != null) throw new IllegalArgumentException("null must be passed to 'nodeType' parameter when 'node' is non-null");
      mySubstrateRef = SubstrateRef.createAstStrongRef(node);
      myElementType = node.getElementType();
    }
  }


  /**
   * Ensures this element is AST-based. This is an expensive operation that might take significant time and allocate lots of objects,
   * so it should be to be avoided if possible.
   *
   * @return an AST node corresponding to this element. If the element is currently operating via stubs,
   * this causes AST to be loaded for the whole file and all stub-based PSI elements in this file (including the current one)
   * to be switched from stub to AST. So, after this call {@link #getStub()} will return null.
   */
  @Override
  public @NotNull ASTNode getNode() {
    if (mySubstrateRef instanceof SubstrateRef.StubRef) {
      ApplicationManager.getApplication().assertReadAccessAllowed();
      PsiFileImpl file = (PsiFileImpl)getContainingFile();
      if (!file.isValid()) throw new PsiInvalidElementAccessException(file);

      FileElement treeElement = file.getTreeElement();
      if (treeElement != null && mySubstrateRef instanceof SubstrateRef.StubRef) {
        return notBoundInExistingAst(file, treeElement);
      }

      treeElement = file.calcTreeElement();
      if (mySubstrateRef instanceof SubstrateRef.StubRef) {
        return failedToBindStubToAst(file, treeElement);
      }
    }

    return mySubstrateRef.getNode();
  }

  private ASTNode failedToBindStubToAst(@NotNull PsiFileImpl file, final @NotNull FileElement fileElement) {
    VirtualFile vFile = file.getVirtualFile();
    StubTree stubTree = file.getStubTree();
    final String stubString = stubTree != null ? ((PsiFileStubImpl<?>)stubTree.getRoot()).printTree() : null;
    final String astString = RecursionManager.doPreventingRecursion("failedToBindStubToAst", true,
                                                                    () -> DebugUtil.treeToString(fileElement, false));

    final @NonNls String message = "Failed to bind stub to AST for element " + getClass() + " in " +
                                   (vFile == null ? "<unknown file>" : vFile.getPath()) +
                                   "\nFile:\n" + file + "@" + System.identityHashCode(file);

    final String creationTraces = ourTraceStubAstBinding ? dumpCreationTraces(fileElement) : null;

    List<Attachment> attachments = new ArrayList<>();
    if (stubString != null) {
      attachments.add(new Attachment("stubTree.txt", stubString));
    }
    if (astString != null) {
      attachments.add(new Attachment("ast.txt", astString));
    }
    if (creationTraces != null) {
      attachments.add(new Attachment("creationTraces.txt", creationTraces));
    }

    throw new RuntimeExceptionWithAttachments(message, attachments.toArray(Attachment.EMPTY_ARRAY));
  }

  private @NotNull String dumpCreationTraces(@NotNull FileElement fileElement) {
    @NonNls StringBuilder traces = new StringBuilder("\nNow " + Thread.currentThread() + "\n");
    traces.append("My creation trace:\n").append(getUserData(CREATION_TRACE));
    traces.append("AST creation traces:\n");
    fileElement.acceptTree(new RecursiveTreeElementWalkingVisitor(false) {
      @Override
      public void visitComposite(CompositeElement composite) {
        PsiElement psi = composite.getPsi();
        if (psi != null) {
          traces.append(psi).append("@").append(System.identityHashCode(psi)).append("\n");
          String trace = psi.getUserData(CREATION_TRACE);
          if (trace != null) {
            traces.append(trace).append("\n");
          }
        }
        super.visitComposite(composite);
      }
    });
    return traces.toString();
  }

  @SuppressWarnings({"NonConstantStringShouldBeStringBuffer", "StringConcatenationInLoop"})
  private ASTNode notBoundInExistingAst(@NotNull PsiFileImpl file, @NotNull FileElement treeElement) {
    @NonNls String message = "file=" + file + "; tree=" + treeElement;
    PsiElement each = this;
    while (each != null) {
      message += "\n each of class " + each.getClass() + "; valid=" + each.isValid();
      if (each instanceof StubBasedPsiElementBase) {
        message += "; ref=" + ((StubBasedPsiElementBase<?>)each).mySubstrateRef;
        each = ((StubBasedPsiElementBase<?>)each).getParentByStub();
      }
      else {
        if (each instanceof PsiFile) {
          message += "; same file=" + (each == file) + "; current tree= " + file.getTreeElement() + "; stubTree=" + file.getStubTree() + "; physical=" + file.isPhysical();
        }
        break;
      }
    }
    StubElement<?> eachStub = getStub();
    while (eachStub != null) {
      message += "\n each stub " + (eachStub instanceof PsiFileStubImpl ? ((PsiFileStubImpl<?>)eachStub).getDiagnostics() : eachStub);
      eachStub = eachStub.getParentStub();
    }

    if (ourTraceStubAstBinding) {
      message += dumpCreationTraces(treeElement);
    }
    throw new AssertionError(message);
  }

  /**
   * Don't invoke this method, it's public for implementation reasons.
   */
  public final void setNode(@NotNull ASTNode node) {
    setSubstrateRef(SubstrateRef.createAstStrongRef(node));
  }

  /**
   * Don't invoke this method, it's public for implementation reasons.
   */
  public final void setSubstrateRef(@NotNull SubstrateRef substrateRef) {
    mySubstrateRef = substrateRef;
  }

  @Override
  public @NotNull Language getLanguage() {
    return myElementType.getLanguage();
  }

  @Override
  public @NotNull PsiFile getContainingFile() {
    try {
      return mySubstrateRef.getContainingFile();
    }
    catch (PsiInvalidElementAccessException e) {
      if (PsiInvalidElementAccessException.getInvalidationTrace(this) != null) {
        throw new PsiInvalidElementAccessException(this, e);
      }
      throw e;
    }
  }

  @Override
  public boolean isWritable() {
    return getContainingFile().isWritable();
  }

  @Override
  public boolean isValid() {
    ProgressManager.checkCanceled();
    return mySubstrateRef.isValid();
  }

  @Override
  public PsiManagerEx getManager() {
    Project project = ProjectCoreUtil.theOnlyOpenProject();
    if (project != null) {
      return PsiManagerEx.getInstanceEx(project);
    }
    return (PsiManagerEx)getContainingFile().getManager();
  }

  @Override
  public @NotNull Project getProject() {
    Project project = ProjectCoreUtil.theOnlyOpenProject();
    if (project != null) {
      return project;
    }
    return getContainingFile().getProject();
  }

  @Override
  public boolean isPhysical() {
    return getContainingFile().isPhysical();
  }

  @Override
  public PsiElement getContext() {
    T stub = getStub();
    if (stub != null) {
      if (!(stub instanceof PsiFileStub)) {
        return stub.getParentStub().getPsi();
      }
    }
    return super.getContext();
  }

  /**
   * Please consider using {@link #getParent()} instead, because this method can return different results before and after AST is loaded.
   * @return a PSI element taken from parent stub (if present) or parent AST node.
   */
  protected final PsiElement getParentByStub() {
    final StubElement<?> stub = getStub();
    if (stub != null) {
      return stub.getParentStub().getPsi();
    }

    return SharedImplUtil.getParent(getNode());
  }

  /**
   * @return the parent of this element. Uses stub hierarchy if possible, but might cause an expensive switch to AST
   * if the parent stub doesn't correspond to the parent AST node.
   */
  @Override
  public PsiElement getParent() {
    T stub = getGreenStub();
    if (stub != null && !((ObjectStubBase<?>)stub).isDangling()) {
      return stub.getParentStub().getPsi();
    }

    return SourceTreeToPsiMap.treeElementToPsi(getNode().getTreeParent());
  }

  public @NotNull IStubElementType getElementType() {
    if (!(myElementType instanceof IStubElementType)) {
      throw new ClassCastException("Not a stub type: " + myElementType + " in " + getClass());
    }
    return (IStubElementType<?, ?>)myElementType;
  }

  /**
   * Note: for most clients (where the logic doesn't crucially differ for stub and AST cases), {@link #getGreenStub()} should be preferred.
   * @return the stub that this element is built upon, or null if the element is currently AST-based. The latter can happen
   * if the file text was loaded from the very beginning, or if it was loaded via {@link #getNode()} on this or any other element
   * in the containing file.
   */
  public @Nullable T getStub() {
    ProgressIndicatorProvider.checkCanceled(); // Hope, this is called often
    //noinspection unchecked
    return (T)mySubstrateRef.getStub();
  }

  /**
   * Like {@link #getStub()}, but can return a non-null value after the element has been switched to AST. Can be used
   * to retrieve the information which is cheaper to get from a stub than by tree traversal.
   * @see PsiFileImpl#getGreenStub()
   */
  public final @Nullable T getGreenStub() {
    ProgressIndicatorProvider.checkCanceled(); // Hope, this is called often
    //noinspection unchecked
    return (T)mySubstrateRef.getGreenStub();
  }

  /**
   * @return a child of specified type, taken from stubs (if this element is currently stub-based) or AST (otherwise).
   */
  public @Nullable <Psi extends PsiElement> Psi getStubOrPsiChild(@NotNull IStubElementType<? extends StubElement, Psi> elementType) {
    T stub = getGreenStub();
    if (stub != null) {
      //noinspection unchecked
      final StubElement<Psi> element = stub.findChildStubByType(elementType);
      if (element != null) {
        return element.getPsi();
      }
    }
    else {
      final ASTNode childNode = getNode().findChildByType(elementType);
      if (childNode != null) {
        //noinspection unchecked
        return (Psi)childNode.getPsi();
      }
    }
    return null;
  }

  /**
   * @return a not-null child of specified type, taken from stubs (if this element is currently stub-based) or AST (otherwise).
   */
  public @NotNull <S extends StubElement<?>, Psi extends PsiElement> Psi getRequiredStubOrPsiChild(@NotNull IStubElementType<S, Psi> elementType) {
    Psi result = getStubOrPsiChild(elementType);
    if (result == null) {
      throw new AssertionError("Missing required child of type " + elementType + "; tree: " + DebugUtil.psiToString(this, true));
    }
    return result;
  }

  /**
   * @return children of specified type, taken from stubs (if this element is currently stub-based) or AST (otherwise).
   */
  public <S extends StubElement<?>, Psi extends PsiElement> Psi @NotNull [] getStubOrPsiChildren(@NotNull IStubElementType<S, ? extends Psi> elementType, Psi @NotNull [] array) {
    T stub = getGreenStub();
    if (stub != null) {
      //noinspection unchecked
      return (Psi[])stub.getChildrenByType(elementType, array);
    }
    else {
      final ASTNode[] nodes = SharedImplUtil.getChildrenOfType(getNode(), elementType);
      Psi[] psiElements = ArrayUtil.newArray(ArrayUtil.getComponentType(array), nodes.length);
      for (int i = 0; i < nodes.length; i++) {
        //noinspection unchecked
        psiElements[i] = (Psi)nodes[i].getPsi();
      }
      return psiElements;
    }
  }

  /**
   * @return children of specified type, taken from stubs (if this element is currently stub-based) or AST (otherwise).
   */
  public <S extends StubElement<?>, Psi extends PsiElement> Psi @NotNull [] getStubOrPsiChildren(@NotNull IStubElementType<S, ? extends Psi> elementType, @NotNull ArrayFactory<? extends Psi> f) {
    T stub = getGreenStub();
    if (stub != null) {
      //noinspection unchecked
      return (Psi[])stub.getChildrenByType(elementType, f);
    }
    else {
      final ASTNode[] nodes = SharedImplUtil.getChildrenOfType(getNode(), elementType);
      Psi[] psiElements = f.create(nodes.length);
      for (int i = 0; i < nodes.length; i++) {
        //noinspection unchecked
        psiElements[i] = (Psi)nodes[i].getPsi();
      }
      return psiElements;
    }
  }

  /**
   * @return children of specified type, taken from stubs (if this element is currently stub-based) or AST (otherwise).
   */
  public <Psi extends PsiElement> Psi @NotNull [] getStubOrPsiChildren(@NotNull TokenSet filter, Psi @NotNull [] array) {
    T stub = getGreenStub();
    if (stub != null) {
      //noinspection unchecked
      return (Psi[])stub.getChildrenByType(filter, array);
    }
    else {
      final ASTNode[] nodes = getNode().getChildren(filter);
      Psi[] psiElements = ArrayUtil.newArray(ArrayUtil.getComponentType(array), nodes.length);
      for (int i = 0; i < nodes.length; i++) {
        //noinspection unchecked
        psiElements[i] = (Psi)nodes[i].getPsi();
      }
      return psiElements;
    }
  }

  /**
   * @return children of specified type, taken from stubs (if this element is currently stub-based) or AST (otherwise).
   */
  public <Psi extends PsiElement> Psi @NotNull [] getStubOrPsiChildren(@NotNull TokenSet filter, @NotNull ArrayFactory<? extends Psi> f) {
    T stub = getGreenStub();
    if (stub != null) {
      //noinspection unchecked
      return (Psi[])stub.getChildrenByType(filter, f);
    }
    else {
      final ASTNode[] nodes = getNode().getChildren(filter);
      Psi[] psiElements = f.create(nodes.length);
      for (int i = 0; i < nodes.length; i++) {
        //noinspection unchecked
        psiElements[i] = (Psi)nodes[i].getPsi();
      }
      return psiElements;
    }
  }

  /**
   * @return a first ancestor of specified type, in stub hierarchy (if this element is currently stub-based) or AST hierarchy (otherwise).
   */
  protected @Nullable <E extends PsiElement> E getStubOrPsiParentOfType(@NotNull Class<E> parentClass) {
    T stub = getStub();
    if (stub != null) {
      //noinspection unchecked
      return (E)stub.getParentStubOfType(parentClass);
    }
    return PsiTreeUtil.getParentOfType(this, parentClass);
  }

  @Override
  protected Object clone() {
    final StubBasedPsiElementBase<?> copy = (StubBasedPsiElementBase<?>)super.clone();
    copy.setSubstrateRef(SubstrateRef.createAstStrongRef(getNode()));
    return copy;
  }
}
