// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.openapi.diagnostic

import ksp.com.intellij.openapi.progress.ProcessCanceledException
import ksp.org.jetbrains.annotations.NonNls
import java.util.concurrent.CancellationException

inline fun <reified T : Any> T.thisLogger(): Logger = Logger.getInstance(T::class.java)

inline fun <reified T : Any> logger(): Logger = Logger.getInstance(T::class.java)

inline fun Logger.debug(e: Exception? = null, lazyMessage: () -> @NonNls String) {
  if (isDebugEnabled) {
    debug(lazyMessage(), e)
  }
}

inline fun Logger.trace(@NonNls lazyMessage: () -> String) {
  if (isTraceEnabled) {
    trace(lazyMessage())
  }
}

/** Consider using [Result.getOrLogException] for more straight-forward API instead. */
inline fun <T> Logger.runAndLogException(runnable: () -> T): T? {
  return runCatching {
    runnable()
  }.getOrLogException(this)
}

fun <T> Result<T>.getOrLogException(logger: Logger): T? {
  return onFailure { e ->
    if (e is ProcessCanceledException || e is CancellationException) {
      throw e
    }
    else {
      logger.error(e)
    }
  }.getOrNull()
}