// Copyright 2000-2021 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.psi.presentation.java;

import ksp.com.intellij.core.JavaPsiBundle;
import ksp.com.intellij.navigation.ColoredItemPresentation;
import ksp.com.intellij.navigation.ItemPresentation;
import ksp.com.intellij.navigation.ItemPresentationProvider;
import ksp.com.intellij.openapi.editor.colors.CodeInsightColors;
import ksp.com.intellij.openapi.editor.colors.TextAttributesKey;
import ksp.com.intellij.openapi.project.IndexNotReadyException;
import ksp.com.intellij.openapi.util.Iconable;
import ksp.com.intellij.psi.PsiClass;
import ksp.com.intellij.psi.PsiClassOwner;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.ui.NewUiValue;
import ksp.org.jetbrains.annotations.NotNull;

import javax.swing.*;


public class ClassPresentationProvider implements ItemPresentationProvider<PsiClass> {
  @Override
  public ItemPresentation getPresentation(@NotNull final PsiClass psiClass) {
    return new ColoredItemPresentation() {
      @Override
      public String getPresentableText() {
        return ClassPresentationUtil.getNameForClass(psiClass, false);
      }

      @Override
      public String getLocationString() {
        PsiFile file = psiClass.getContainingFile();
        if (file instanceof PsiClassOwner) {
          PsiClassOwner classOwner = (PsiClassOwner)file;
          String packageName = classOwner.getPackageName();
          if (packageName.isEmpty()) return null;
          return NewUiValue.isEnabled() ? JavaPsiBundle.message("aux.context.display", packageName) : "(" + packageName + ")";
        }
        return null;
      }

      @Override
      public TextAttributesKey getTextAttributesKey() {
        try {
          if (psiClass.isDeprecated()) {
            return CodeInsightColors.DEPRECATED_ATTRIBUTES;
          }
        }
        catch (IndexNotReadyException ignore) {
        }
        return null;
      }

      @Override
      public Icon getIcon(boolean open) {
        return psiClass.getIcon(Iconable.ICON_FLAG_VISIBILITY | Iconable.ICON_FLAG_READ_STATUS);
      }
    };
  }
}
