// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.psi.util;

import ksp.com.intellij.openapi.util.text.StringUtil;
import ksp.com.intellij.psi.PsiClass;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.psi.PsiJavaFile;
import ksp.com.intellij.psi.PsiUnnamedClass;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

public final class JavaUnnamedClassUtil {
  public static boolean isFileWithUnnamedClass(@NotNull PsiFile file) {
    if (file instanceof PsiJavaFile) {
      PsiJavaFile javaFile = (PsiJavaFile)file;
      if (getUnnamedClassFor(javaFile) != null) return true;
    }
    return false;
  }

  /**
   * Retrieves the unnamed class PSI element from the given PsiFile.
   *
   * @param file the PsiFile from which to retrieve the unnamed class
   * @return the unnamed class if found, null otherwise
   */
  @Nullable
  public static PsiUnnamedClass getUnnamedClassFor(@NotNull PsiFile file) {
    if (file instanceof PsiJavaFile) {
      final PsiJavaFile javaFile = (PsiJavaFile)file;
      PsiClass[] classes = javaFile.getClasses();
      if (classes.length == 1 && classes[0] instanceof PsiUnnamedClass) {
        return (PsiUnnamedClass)classes[0];
      }
    }
    return null;
  }

  /**
   * @return JVM name if the file name is a valid class identifier (e.g., contains no space)
   */
  public static @Nullable String getJvmName(@NotNull PsiUnnamedClass unnamedClass) {
    PsiFile containingFile = unnamedClass.getContainingFile();
    if (containingFile == null) return null;
    String name = containingFile.getName();
    int extensionIndex = name.lastIndexOf(".java");
    if (extensionIndex == -1) return null;
    String jvmName = name.substring(0, extensionIndex);
    if (!StringUtil.isJavaIdentifier(jvmName)) return null;
    return jvmName;
  }

  /**
   * @param name the name of the unnamed class (might include the ".java" extension)
   * @return the JVM name of the unnamed class
   */
  public static String getJvmName(@NotNull String name) {
    return StringUtil.trimEnd(name, ".java", true);
  }
}
