// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.util.concurrency;

import ksp.com.intellij.diagnostic.ThreadDumper;
import ksp.com.intellij.openapi.application.ApplicationManager;
import ksp.com.intellij.openapi.diagnostic.Attachment;
import ksp.com.intellij.openapi.diagnostic.Logger;
import ksp.com.intellij.openapi.diagnostic.RuntimeExceptionWithAttachments;
import ksp.com.intellij.util.ui.EDT;
import ksp.org.jetbrains.annotations.ApiStatus.Internal;
import ksp.org.jetbrains.annotations.ApiStatus.Obsolete;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;
import ksp.org.jetbrains.annotations.VisibleForTesting;

import java.awt.*;

/**
 * This class contains various threading assertions.
 * Calls to these functions are generated by {@link org.jetbrains.jps.devkit.threadingModelHelper.TMHInstrumenter TMHInstrumenter}
 * if a function is annotated with a respective annotation and annotation's {@code generateAssertion} is set to {@code true}.
 * These functions are also allowed to be used directly.
 */
public final class ThreadingAssertions {

  private ThreadingAssertions() { }

  private static @NotNull Logger getLogger() {
    return Logger.getInstance(ThreadingAssertions.class);
  }

  @Internal
  @VisibleForTesting
  public static final String MUST_EXECUTE_INSIDE_READ_ACTION =
    "Read access is allowed from inside read-action (see Application.runReadAction())";
  @Internal
  @VisibleForTesting
  public static final String MUST_NOT_EXECUTE_INSIDE_READ_ACTION =
    "Must not execute inside read action";
  private static final String MUST_EXECUTE_IN_WRITE_INTENT_READ_ACTION =
    "Access is allowed from write thread only";
  @Internal
  @VisibleForTesting
  public static final String MUST_EXECUTE_INSIDE_WRITE_ACTION =
    "Write access is allowed inside write-action only (see Application.runWriteAction())";
  @Internal
  @VisibleForTesting
  public static final String MUST_EXECUTE_UNDER_EDT =
    "Access is allowed from Event Dispatch Thread (EDT) only";
  @Internal
  @VisibleForTesting
  public static final String MUST_NOT_EXECUTE_UNDER_EDT =
    "Access from Event Dispatch Thread (EDT) is not allowed";

  private static final String DOCUMENTATION_URL = "https://jb.gg/ij-platform-threading";

  /**
   * Asserts that the current thread is the event dispatch thread.
   *
   * @see com.intellij.util.concurrency.annotations.RequiresEdt
   */
  public static void assertEventDispatchThread() {
    if (!EDT.isCurrentThreadEdt()) {
      throwThreadAccessException(MUST_EXECUTE_UNDER_EDT);
    }
  }

  /**
   * Asserts that the current thread is <b>not</b> the event dispatch thread.
   *
   * @see com.intellij.util.concurrency.annotations.RequiresBackgroundThread
   */
  public static void assertBackgroundThread() {
    if (EDT.isCurrentThreadEdt()) {
      throwThreadAccessException(MUST_NOT_EXECUTE_UNDER_EDT);
    }
  }

  /**
   * Asserts that the current thread has read access.
   * <p/>
   * For consistency with other assertions, this function <b>throws</b> an error when called without holding the read lock.
   *
   * @see #softAssertReadAccess
   * @see com.intellij.util.concurrency.annotations.RequiresReadLock
   */
  public static void assertReadAccess() {
    if (!ApplicationManager.getApplication().isReadAccessAllowed()) {
      throwThreadAccessException(MUST_EXECUTE_INSIDE_READ_ACTION);
    }
  }

  /**
   * Asserts that the current thread has read access <b>without throwing</b>.
   * <p/>
   * Historically, it was not possible to throw everywhere,
   * so this function logs an error without throwing, but then proceeds normally.
   * When writing the new code, please prefer throwing {@link #assertReadAccess} instead,
   * because only it can guarantee that the caller holds the read lock.
   *
   * @see com.intellij.util.concurrency.annotations.RequiresReadLock
   */
  @Obsolete
  public static void softAssertReadAccess() {
    if (!ApplicationManager.getApplication().isReadAccessAllowed()) {
      getLogger().error(createThreadAccessException(MUST_EXECUTE_INSIDE_READ_ACTION));
    }
  }

  /**
   * Asserts that the current thread has <b>no</b> read access.
   *
   * @see com.intellij.util.concurrency.annotations.RequiresReadLockAbsence
   */
  public static void assertNoReadAccess() {
    if (ApplicationManager.getApplication().isReadAccessAllowed()) {
      throwThreadAccessException(MUST_NOT_EXECUTE_INSIDE_READ_ACTION);
    }
  }

  /**
   * Asserts that the current thread has write-intent read access.
   */
  public static void assertWriteIntentReadAccess() {
    if (!ApplicationManager.getApplication().isWriteIntentLockAcquired()) {
      throwThreadAccessException(MUST_EXECUTE_IN_WRITE_INTENT_READ_ACTION);
    }
  }

  /**
   * Asserts that the current thread has write access.
   *
   * @see com.intellij.util.concurrency.annotations.RequiresWriteLock
   */
  public static void assertWriteAccess() {
    if (!ApplicationManager.getApplication().isWriteAccessAllowed()) {
      throwThreadAccessException(MUST_EXECUTE_INSIDE_WRITE_ACTION);
    }
  }

  private static void throwThreadAccessException(@NotNull @NonNls String message) {
    throw createThreadAccessException(message);
  }

  private static @NotNull RuntimeExceptionWithAttachments createThreadAccessException(@NonNls @NotNull String message) {
    return new RuntimeExceptionWithAttachments(
      message + "; see " + DOCUMENTATION_URL + " for details" + "\n" + getThreadDetails(),
      new Attachment("threadDump.txt", ThreadDumper.dumpThreadsToString())
    );
  }

  private static @NotNull String getThreadDetails() {
    Thread current = Thread.currentThread();
    Thread edt = EDT.getEventDispatchThread();
    return "Current thread: " + describe(current) + " (EventQueue.isDispatchThread()=" + EventQueue.isDispatchThread() + ")\n" +
           "SystemEventQueueThread: " + (edt == current ? "(same)" : describe(edt));
  }

  private static @NotNull String describe(@Nullable Thread o) {
    return o == null ? "null" : o + " " + System.identityHashCode(o);
  }
}
