/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotation
import ksp.org.jetbrains.kotlin.analysis.api.fir.annotations.computeAnnotationArguments
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.annotations.KaAnnotationImpl
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSymbol
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.sessions.LLFirSession
import ksp.org.jetbrains.kotlin.builtins.StandardNames
import ksp.org.jetbrains.kotlin.descriptors.annotations.AnnotationUseSiteTarget
import ksp.org.jetbrains.kotlin.fir.FirAnnotationContainer
import ksp.org.jetbrains.kotlin.fir.FirElement
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.FirResolvePhase
import ksp.org.jetbrains.kotlin.fir.declarations.getAnnotationsByClassId
import ksp.org.jetbrains.kotlin.fir.declarations.getStringArgument
import ksp.org.jetbrains.kotlin.fir.declarations.toAnnotationClassId
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isOverride
import ksp.org.jetbrains.kotlin.fir.diagnostics.ConeDiagnostic
import ksp.org.jetbrains.kotlin.fir.diagnostics.ConeUnreportedDuplicateDiagnostic
import ksp.org.jetbrains.kotlin.fir.expressions.*
import ksp.org.jetbrains.kotlin.fir.psi
import ksp.org.jetbrains.kotlin.fir.references.*
import ksp.org.jetbrains.kotlin.fir.resolve.diagnostics.ConeDiagnosticWithCandidates
import ksp.org.jetbrains.kotlin.fir.resolve.diagnostics.ConeDiagnosticWithSymbol
import ksp.org.jetbrains.kotlin.fir.resolve.diagnostics.ConeHiddenCandidateError
import ksp.org.jetbrains.kotlin.fir.resolve.scope
import ksp.org.jetbrains.kotlin.fir.resolve.toClassSymbol
import ksp.org.jetbrains.kotlin.fir.scopes.CallableCopyTypeCalculator
import ksp.org.jetbrains.kotlin.fir.scopes.getDeclaredConstructors
import ksp.org.jetbrains.kotlin.fir.scopes.getDirectOverriddenProperties
import ksp.org.jetbrains.kotlin.fir.scopes.unsubstitutedScope
import ksp.org.jetbrains.kotlin.fir.symbols.FirBasedSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirConstructorSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirPropertyAccessorSymbol
import ksp.org.jetbrains.kotlin.name.JvmStandardClassIds
import ksp.org.jetbrains.kotlin.psi.KtCallElement
import ksp.org.jetbrains.kotlin.util.OperatorNameConventions

/**
 * Returns `true` if the symbol is for a function named `invoke`.
 */
internal fun FirBasedSymbol<*>.isInvokeFunction() =
    (this as? FirNamedFunctionSymbol)?.fir?.name == OperatorNameConventions.INVOKE

internal fun FirFunctionCall.getCalleeSymbol(): FirBasedSymbol<*>? =
    calleeReference.getResolvedSymbolOfNameReference()

internal fun FirFunctionCall.getCandidateSymbols(): Collection<FirBasedSymbol<*>> =
    calleeReference.getCandidateSymbols()

internal fun FirReference.getResolvedSymbolOfNameReference(): FirBasedSymbol<*>? =
    (this as? FirResolvedNamedReference)?.resolvedSymbol

internal fun FirReference.getResolvedKtSymbolOfNameReference(builder: KaSymbolByFirBuilder): KaSymbol? =
    getResolvedSymbolOfNameReference()?.fir?.let(builder::buildSymbol)

internal fun FirErrorNamedReference.getCandidateSymbols(): Collection<FirBasedSymbol<*>> =
    diagnostic.getCandidateSymbols()

internal fun FirNamedReference.getCandidateSymbols(): Collection<FirBasedSymbol<*>> = when (this) {
    is FirResolvedNamedReference -> listOf(resolvedSymbol)
    is FirErrorNamedReference -> getCandidateSymbols()
    else -> emptyList()
}

internal fun ConeDiagnostic.getCandidateSymbols(): Collection<FirBasedSymbol<*>> =
    when (this) {
        is ConeHiddenCandidateError -> {
            // Candidate with @Deprecated(DeprecationLevel.HIDDEN)
            emptyList()
        }
        is ConeDiagnosticWithCandidates -> candidateSymbols
        is ConeDiagnosticWithSymbol<*> -> listOf(symbol)
        is ConeUnreportedDuplicateDiagnostic -> original.getCandidateSymbols()
        else -> emptyList()
    }

internal fun FirAnnotation.toKaAnnotation(builder: KaSymbolByFirBuilder): KaAnnotation {
    val constructorSymbol = findAnnotationConstructor(this, builder.rootSession)
        ?.let(builder.functionBuilder::buildConstructorSymbol)

    val classId = toAnnotationClassId(builder.rootSession)

    return KaAnnotationImpl(
        classId = classId,
        psi = psi as? KtCallElement,
        useSiteTarget = useSiteTarget,
        lazyArguments = if (this !is FirAnnotationCall || arguments.isNotEmpty())
            lazy { computeAnnotationArguments(this, builder) }
        else
            lazyOf(emptyList()),
        constructorSymbol = constructorSymbol,
        token = builder.token,
    )
}

private fun findAnnotationConstructor(annotation: FirAnnotation, session: LLFirSession): FirConstructorSymbol? {
    if (annotation is FirAnnotationCall) {
        val constructorSymbol = annotation.calleeReference.toResolvedConstructorSymbol()
        if (constructorSymbol != null) {
            return constructorSymbol
        }
    }

    // Handle unresolved annotation calls gracefully
    @OptIn(UnresolvedExpressionTypeAccess::class)
    val annotationClass = annotation.coneTypeOrNull?.toClassSymbol(session)?.fir ?: return null

    // The search is done via scope to force Java enhancement. Annotation class might be a 'FirJavaClass'
    return annotationClass
        .unsubstitutedScope(session, session.getScopeSession(), withForcedTypeCalculator = false, memberRequiredPhase = null)
        .getDeclaredConstructors()
        .singleOrNull()
}

/**
 * Implicit dispatch receiver is present when an extension function declared in object
 * is imported somewhere else and used without directly referencing the object instance
 * itself:
 *
 * ```kt
 * import Foo.bar
 *
 * object Foo { fun String.bar() {} }
 *
 * fun usage() {
 *   "hello".bar() // this call has implicit 'Foo' dispatch receiver
 * }
 * ```
 */
internal val FirResolvedQualifier.isImplicitDispatchReceiver: Boolean
    get() = source?.kind == KtFakeSourceElementKind.ImplicitReceiver

internal fun FirAnnotationContainer.getJvmNameFromAnnotation(session: FirSession, target: AnnotationUseSiteTarget? = null): String? {
    val annotationCalls = getAnnotationsByClassId(JvmStandardClassIds.Annotations.JvmName, session)
    return annotationCalls.firstNotNullOfOrNull { call ->
        call.getStringArgument(StandardNames.NAME, session)
            ?.takeIf { target == null || call.useSiteTarget == target }
    }
}

internal fun FirElement.unwrapSafeCall(): FirElement =
    (this as? FirSafeCallExpression)?.selector ?: this

internal fun FirPropertyAccessorSymbol.isSetterOverride(analysisSession: KaFirSession): Boolean {
    if (isGetter) return false
    if (isOverride) return true

    val propertySymbol = fir.propertySymbol
    if (!propertySymbol.isOverride) return false
    val session = analysisSession.firSession
    val containingClassScope = dispatchReceiverType?.scope(
        session,
        analysisSession.getScopeSessionFor(session),
        CallableCopyTypeCalculator.DoNothing,
        requiredMembersPhase = FirResolvePhase.STATUS,
    ) ?: return false

    val overriddenProperties = containingClassScope.getDirectOverriddenProperties(propertySymbol)
    return overriddenProperties.any { it.isVar }
}