/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.components

import ksp.org.jetbrains.kotlin.analysis.api.components.KaBuiltinTypes
import ksp.org.jetbrains.kotlin.analysis.api.components.KaTypeProvider
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.KaFirSymbol
import ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.dispatchReceiverType
import ksp.org.jetbrains.kotlin.analysis.api.fir.types.KaFirType
import ksp.org.jetbrains.kotlin.analysis.api.fir.types.PublicTypeApproximator
import ksp.org.jetbrains.kotlin.analysis.api.fir.utils.ConeSupertypeCalculationMode
import ksp.org.jetbrains.kotlin.analysis.api.fir.utils.firSymbol
import ksp.org.jetbrains.kotlin.analysis.api.fir.utils.getAllStrictSupertypes
import ksp.org.jetbrains.kotlin.analysis.api.fir.utils.getDirectSupertypes
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.components.KaSessionComponent
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaAnonymousObjectSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaCallableSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaClassLikeSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaClassifierSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaNamedClassSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaReceiverParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaTypeAliasSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaTypeParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.types.KaType
import ksp.org.jetbrains.kotlin.analysis.api.types.KaTypeNullability
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.api.*
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.sessions.llFirSession
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.util.ContextCollector
import ksp.org.jetbrains.kotlin.fir.FirElement
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.ConeTypeCompatibilityChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.ConeTypeCompatibilityChecker.isCompatible
import ksp.org.jetbrains.kotlin.fir.declarations.FirMemberDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirProperty
import ksp.org.jetbrains.kotlin.fir.declarations.FirResolvePhase
import ksp.org.jetbrains.kotlin.fir.expressions.*
import ksp.org.jetbrains.kotlin.fir.java.enhancement.EnhancedForWarningConeSubstitutor
import ksp.org.jetbrains.kotlin.fir.psi
import ksp.org.jetbrains.kotlin.fir.resolve.SessionHolderImpl
import ksp.org.jetbrains.kotlin.fir.resolve.defaultType
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirCallableSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirClassLikeSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirClassSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirTypeAliasSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirTypeParameterSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirValueParameterSymbol
import ksp.org.jetbrains.kotlin.fir.types.*
import ksp.org.jetbrains.kotlin.fir.types.impl.ConeClassLikeTypeImpl
import ksp.org.jetbrains.kotlin.fir.types.impl.ConeTypeParameterTypeImpl
import ksp.org.jetbrains.kotlin.fir.utils.exceptions.withFirSymbolEntry
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.psi.*
import ksp.org.jetbrains.kotlin.utils.exceptions.errorWithAttachment
import ksp.org.jetbrains.kotlin.utils.exceptions.withPsiEntry

internal class KaFirTypeProvider(
    override val analysisSessionProvider: () -> KaFirSession,
) : KaSessionComponent<KaFirSession>(), KaTypeProvider, KaFirSessionComponent {
    override val builtinTypes: KaBuiltinTypes by lazy {
        KaFirBuiltInTypes(rootModuleSession.builtinTypes, firSymbolBuilder, token)
    }

    override fun KaType.approximateToSuperPublicDenotable(approximateLocalTypes: Boolean): KaType? = withValidityAssertion {
        require(this is KaFirType)
        val coneType = coneType
        val approximatedConeType = PublicTypeApproximator.approximateTypeToPublicDenotable(
            coneType,
            rootModuleSession,
            approximateLocalTypes = approximateLocalTypes,
        )

        return approximatedConeType?.asKtType()
    }

    override fun KaType.approximateToSubPublicDenotable(approximateLocalTypes: Boolean): KaType? = withValidityAssertion {
        require(this is KaFirType)
        val coneType = coneType
        val approximatedConeType = rootModuleSession.typeApproximator.approximateToSubType(
            coneType,
            PublicTypeApproximator.PublicApproximatorConfiguration(approximateLocalTypes = approximateLocalTypes),
        )

        return approximatedConeType?.asKtType()
    }

    override val KaType.enhancedType: KaType?
        get() = withValidityAssertion {
            require(this is KaFirType)
            val coneType = coneType
            val substitutor = EnhancedForWarningConeSubstitutor(typeContext)
            val enhancedConeType = substitutor.substituteType(coneType)

            return enhancedConeType?.asKtType()
        }

    override val KaClassifierSymbol.defaultType: KaType
        get() = withValidityAssertion {
            with(analysisSession) {
                val firSymbol = firSymbol
                val defaultConeType = when (firSymbol) {
                    is FirTypeParameterSymbol -> firSymbol.defaultType
                    is FirClassLikeSymbol<*> -> ConeClassLikeTypeImpl(
                        firSymbol.toLookupTag(),
                        firSymbol.typeParameterSymbols.map {
                            ConeTypeParameterTypeImpl(
                                it.toLookupTag(),
                                isMarkedNullable = false
                            )
                        }.toTypedArray(),
                        isMarkedNullable = false,
                    )

                    else -> errorWithAttachment("Unexpected ${firSymbol::class.simpleName}") {
                        withFirSymbolEntry("symbol", firSymbol)
                    }
                }

                defaultConeType.asKtType()
            }
        }

    override val Iterable<KaType>.commonSupertype: KaType
        get() = withValidityAssertion {
            val coneTypes = map { it.coneType }
            if (coneTypes.isEmpty()) {
                throw IllegalArgumentException("Got no types")
            }

            return analysisSession.firSession.typeContext.commonSuperTypeOrNull(coneTypes)!!.asKtType()
        }

    override val KtTypeReference.type: KaType
        get() = withValidityAssertion {
            val fir = getFirBySymbols() ?: getOrBuildFirOfType<FirElement>(firResolveSession)
            return when (fir) {
                is FirResolvedTypeRef -> fir.coneType.asKtType()
                is FirDelegatedConstructorCall -> fir.constructedTypeRef.coneType.asKtType()
                is FirTypeProjectionWithVariance -> {
                    when (val typeRef = fir.typeRef) {
                        is FirResolvedTypeRef -> typeRef.coneType.asKtType()
                        else -> throwUnexpectedFirElementError(fir, this)
                    }
                }
                else -> throwUnexpectedFirElementError(fir, this)
            }
        }

    /**
     * Try to get fir element for type reference through symbols.
     * When the type is declared in compiled code this is faster than building FIR from decompiled text.
     */
    private fun KtTypeReference.getFirBySymbols(): FirElement? {
        val parent = parent
        return when {
            parent is KtParameter && parent.ownerFunction != null && parent.typeReference === this ->
                parent.resolveToFirSymbolOfTypeSafe<FirValueParameterSymbol>(firResolveSession, FirResolvePhase.TYPES)?.fir?.returnTypeRef

            parent is KtCallableDeclaration && (parent is KtNamedFunction || parent is KtProperty)
                    && (parent.receiverTypeReference === this || parent.typeReference === this) -> {
                val firCallable = parent.resolveToFirSymbolOfTypeSafe<FirCallableSymbol<*>>(
                    firResolveSession, FirResolvePhase.TYPES
                )?.fir
                if (parent.receiverTypeReference === this) {
                    firCallable?.receiverParameter?.typeRef
                } else firCallable?.returnTypeRef
            }

            parent is KtConstructorCalleeExpression && parent.parent is KtAnnotationEntry -> {
                fun getFirDeclaration(annotationEntry: KtAnnotationEntry, ktTypeReference: KtTypeReference): FirMemberDeclaration? {
                    if (annotationEntry.typeReference !== ktTypeReference) return null
                    val declaration = annotationEntry.parent?.parent as? KtNamedDeclaration ?: return null
                    return when {
                        declaration is KtClassOrObject -> declaration.resolveToFirSymbolOfTypeSafe<FirClassLikeSymbol<*>>(
                            firResolveSession, FirResolvePhase.TYPES
                        )?.fir
                        declaration is KtParameter && declaration.ownerFunction != null ->
                            declaration.resolveToFirSymbolOfTypeSafe<FirValueParameterSymbol>(
                                firResolveSession, FirResolvePhase.TYPES
                            )?.fir
                        declaration is KtCallableDeclaration && (declaration is KtNamedFunction || declaration is KtProperty) -> {
                            declaration.resolveToFirSymbolOfTypeSafe<FirCallableSymbol<*>>(
                                firResolveSession, FirResolvePhase.TYPES
                            )?.fir
                        }
                        else -> return null
                    }
                }

                fun FirMemberDeclaration.findAnnotationTypeRef(annotationEntry: KtAnnotationEntry) = annotations.find {
                    it.psi === annotationEntry
                }?.annotationTypeRef

                val annotationEntry = parent.parent as KtAnnotationEntry
                val firDeclaration = getFirDeclaration(annotationEntry, this)
                if (firDeclaration != null) {
                    firDeclaration.findAnnotationTypeRef(annotationEntry) ?: (firDeclaration as? FirProperty)?.run {
                        backingField?.findAnnotationTypeRef(annotationEntry)
                            ?: getter?.findAnnotationTypeRef(annotationEntry)
                            ?: setter?.findAnnotationTypeRef(annotationEntry)
                    }
                } else null
            }
            else -> null
        }
    }

    override val KtDoubleColonExpression.receiverType: KaType?
        get() = withValidityAssertion {
            return when (val fir = getOrBuildFir(firResolveSession)) {
                is FirGetClassCall -> {
                    fir.resolvedType.getReceiverOfReflectionType()?.asKtType()
                }
                is FirCallableReferenceAccess -> {
                    when (val explicitReceiver = fir.explicitReceiver) {
                        is FirThisReceiverExpression -> {
                            explicitReceiver.resolvedType.asKtType()
                        }
                        is FirPropertyAccessExpression -> {
                            explicitReceiver.resolvedType.asKtType()
                        }
                        is FirFunctionCall -> {
                            explicitReceiver.resolvedType.asKtType()
                        }
                        is FirResolvedQualifier -> {
                            explicitReceiver.symbol?.toLookupTag()?.constructType(
                                explicitReceiver.typeArguments.map { it.toConeTypeProjection() }.toTypedArray(),
                                isMarkedNullable = explicitReceiver.isNullableLHSForCallableReference
                            )?.asKtType()
                                ?: fir.resolvedType.getReceiverOfReflectionType()?.asKtType()
                        }
                        else -> {
                            fir.resolvedType.getReceiverOfReflectionType()?.asKtType()
                        }
                    }
                }
                else -> {
                    throwUnexpectedFirElementError(fir, this)
                }
            }
        }

    private fun ConeKotlinType.getReceiverOfReflectionType(): ConeKotlinType? {
        if (this !is ConeClassLikeType) return null
        if (lookupTag.classId.packageFqName != StandardClassIds.BASE_REFLECT_PACKAGE) return null
        return typeArguments.firstOrNull()?.type
    }

    override fun KaType.withNullability(newNullability: KaTypeNullability): KaType = withValidityAssertion {
        require(this is KaFirType)
        return coneType.withNullability(nullable = newNullability == KaTypeNullability.NULLABLE, rootModuleSession.typeContext).asKtType()
    }

    override fun KaType.hasCommonSubtypeWith(that: KaType): Boolean = withValidityAssertion {
        return analysisSession.firSession.typeContext.isCompatible(
            this.coneType,
            that.coneType
        ) == ConeTypeCompatibilityChecker.Compatibility.COMPATIBLE
    }

    override fun collectImplicitReceiverTypes(position: KtElement): List<KaType> = withValidityAssertion {
        val ktFile = position.containingKtFile
        val firFile = ktFile.getOrBuildFirFile(firResolveSession)

        val fileSession = firFile.llFirSession
        val sessionHolder = SessionHolderImpl(fileSession, fileSession.getScopeSession())

        val context = ContextCollector.process(firFile, sessionHolder, position)
            ?: errorWithAttachment("Cannot find context for ${position::class}") {
                withPsiEntry("position", position)
            }

        return context.towerDataContext.implicitReceiverStack.map { it.type.asKtType() }
    }

    override fun KaType.directSupertypes(shouldApproximate: Boolean): Sequence<KaType> = withValidityAssertion {
        require(this is KaFirType)
        return coneType.getDirectSupertypes(
            analysisSession.firSession,
            ConeSupertypeCalculationMode.substitution(shouldApproximate),
        ).map { it.asKtType() }
    }

    override fun KaType.allSupertypes(shouldApproximate: Boolean): Sequence<KaType> = withValidityAssertion {
        require(this is KaFirType)
        return coneType.getAllStrictSupertypes(
            analysisSession.firSession,
            ConeSupertypeCalculationMode.substitution(shouldApproximate),
        ).map { it.asKtType() }
    }

    @Suppress("OVERRIDE_DEPRECATION")
    override val KaCallableSymbol.dispatchReceiverType: KaType?
        get() = withValidityAssertion {
            when (this) {
                is KaReceiverParameterSymbol -> null
                else -> {
                    require(this is KaFirSymbol<*>)
                    val firSymbol = firSymbol
                    check(firSymbol is FirCallableSymbol<*>) {
                        "Fir declaration should be FirCallableDeclaration; instead it was ${firSymbol::class}"
                    }
                    return firSymbol.dispatchReceiverType(analysisSession.firSymbolBuilder)
                }
            }
        }

    override val KaType.arrayElementType: KaType?
        get() = withValidityAssertion {
            require(this is KaFirType)
            return coneType.arrayElementType()?.asKtType()
        }

}

