/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.base.KaContextReceiver
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.fir.getAllowedPsi
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.symbols.pointers.KaCannotCreateSymbolPointerForLocalLibraryDeclarationException
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaAnonymousFunctionSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaReceiverParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaValueParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.types.KaType
import ksp.org.jetbrains.kotlin.descriptors.Visibility
import ksp.org.jetbrains.kotlin.fir.declarations.impl.FirResolvedDeclarationStatusImpl
import ksp.org.jetbrains.kotlin.fir.realPsi
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirAnonymousFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.isExtension
import ksp.org.jetbrains.kotlin.psi.KtFunction
import ksp.org.jetbrains.kotlin.psi.KtFunctionLiteral
import ksp.org.jetbrains.kotlin.psi.KtNamedFunction
import ksp.org.jetbrains.kotlin.psi.psiUtil.isExtensionDeclaration

internal class KaFirAnonymousFunctionSymbol private constructor(
    override val backingPsi: KtFunction?,
    override val analysisSession: KaFirSession,
    override val lazyFirSymbol: Lazy<FirAnonymousFunctionSymbol>,
) : KaAnonymousFunctionSymbol(), KaFirKtBasedSymbol<KtFunction, FirAnonymousFunctionSymbol> {
    init {
        when (backingPsi) {
            null, is KtFunctionLiteral -> {}
            is KtNamedFunction -> {
                require(backingPsi.isAnonymous)
            }
        }
    }

    constructor(declaration: KtFunction, session: KaFirSession) : this(
        backingPsi = declaration,
        lazyFirSymbol = lazyFirSymbol(declaration, session),
        analysisSession = session,
    )

    constructor(symbol: FirAnonymousFunctionSymbol, session: KaFirSession) : this(
        backingPsi = symbol.fir.realPsi as? KtFunction,
        lazyFirSymbol = lazyOf(symbol),
        analysisSession = session,
    )

    override val psi: PsiElement? get() = withValidityAssertion { backingPsi ?: firSymbol.fir.getAllowedPsi() }
    override val annotations: KaAnnotationList get() = withValidityAssertion { psiOrSymbolAnnotationList() }
    override val returnType: KaType get() = withValidityAssertion { createReturnType() }

    override val receiverParameter: KaReceiverParameterSymbol?
        get() = withValidityAssertion {
            KaFirReceiverParameterSymbol.create(
                // We cannot use KtFunctionLiteral as the backing PSI as it doesn't receiver parameter in the code
                backingPsi?.takeIf { it is KtNamedFunction },
                analysisSession,
                this,
            )
        }

    override val contextReceivers: List<KaContextReceiver>
        get() = withValidityAssertion { createContextReceivers() }

    override val compilerVisibility: Visibility
        get() = withValidityAssertion { FirResolvedDeclarationStatusImpl.DEFAULT_STATUS_FOR_STATUSLESS_DECLARATIONS.visibility }

    override val valueParameters: List<KaValueParameterSymbol>
        get() = withValidityAssertion {
            // lambda may have an implicit argument, so we cannot check it by PSI
            if (backingPsi is KtNamedFunction || backingPsi is KtFunctionLiteral && backingPsi.arrow != null) {
                createKaValueParameters()?.let { return it }
            }

            firSymbol.createKtValueParameters(builder)
        }

    override val isExtension: Boolean
        get() = withValidityAssertion { backingPsi?.isExtensionDeclaration() ?: firSymbol.isExtension }

    override fun createPointer(): KaSymbolPointer<KaAnonymousFunctionSymbol> = withValidityAssertion {
        psiBasedSymbolPointerOfTypeIfSource<KaAnonymousFunctionSymbol>()
            ?: throw KaCannotCreateSymbolPointerForLocalLibraryDeclarationException(this::class)
    }

    override fun equals(other: Any?): Boolean = psiOrSymbolEquals(other)
    override fun hashCode(): Int = psiOrSymbolHashCode()
}
