/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.fir.findPsi
import ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.pointers.KaFirConstructorSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.pointers.createOwnerPointer
import ksp.org.jetbrains.kotlin.analysis.api.fir.visibilityByModifiers
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaConstructorSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaTypeParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaValueParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.symbols.typeParameters
import ksp.org.jetbrains.kotlin.analysis.api.types.KaType
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.providers.FirCallableSignature
import ksp.org.jetbrains.kotlin.descriptors.Visibility
import ksp.org.jetbrains.kotlin.fir.containingClassLookupTag
import ksp.org.jetbrains.kotlin.fir.declarations.utils.hasStableParameterNames
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isActual
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isExpect
import ksp.org.jetbrains.kotlin.fir.declarations.utils.visibility
import ksp.org.jetbrains.kotlin.fir.realPsi
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirConstructorSymbol
import ksp.org.jetbrains.kotlin.lexer.KtTokens
import ksp.org.jetbrains.kotlin.name.ClassId
import ksp.org.jetbrains.kotlin.psi.KtConstructor
import ksp.org.jetbrains.kotlin.psi.KtPrimaryConstructor
import ksp.org.jetbrains.kotlin.psi.psiUtil.isExpectDeclaration

internal class KaFirConstructorSymbol private constructor(
    override val backingPsi: KtConstructor<*>?,
    override val analysisSession: KaFirSession,
    override val lazyFirSymbol: Lazy<FirConstructorSymbol>,
) : KaConstructorSymbol(), KaFirKtBasedSymbol<KtConstructor<*>, FirConstructorSymbol> {
    constructor(declaration: KtConstructor<*>, session: KaFirSession) : this(
        backingPsi = declaration,
        lazyFirSymbol = lazyFirSymbol(declaration, session),
        analysisSession = session,
    )

    constructor(symbol: FirConstructorSymbol, session: KaFirSession) : this(
        backingPsi = symbol.fir.realPsi as? KtConstructor<*>,
        lazyFirSymbol = lazyOf(symbol),
        analysisSession = session,
    )

    override val psi: PsiElement? get() = withValidityAssertion { backingPsi ?: firSymbol.findPsi() }

    override val returnType: KaType get() = withValidityAssertion { firSymbol.returnType(builder) }

    override val valueParameters: List<KaValueParameterSymbol>
        get() = withValidityAssertion {
            createKaValueParameters() ?: firSymbol.createKtValueParameters(builder)
        }

    override val hasStableParameterNames: Boolean
        get() = withValidityAssertion {
            if (backingPsi != null) {
                true
            } else {
                firSymbol.fir.hasStableParameterNames
            }
        }

    override val compilerVisibility: Visibility
        get() = withValidityAssertion { backingPsi?.visibilityByModifiers ?: firSymbol.visibility }

    override val annotations: KaAnnotationList
        get() = withValidityAssertion {
            psiOrSymbolAnnotationList()
        }

    override val containingClassId: ClassId?
        get() = withValidityAssertion {
            backingPsi?.getContainingClassOrObject()?.getClassId()
                ?: firSymbol.containingClassLookupTag()?.classId?.takeUnless { it.isLocal }
        }

    override val isPrimary: Boolean
        get() = withValidityAssertion {
            if (backingPsi != null) {
                backingPsi is KtPrimaryConstructor
            } else {
                firSymbol.isPrimary
            }
        }

    override val isActual: Boolean
        get() = withValidityAssertion { backingPsi?.hasModifier(KtTokens.ACTUAL_KEYWORD) ?: firSymbol.isActual }

    override val isExpect: Boolean
        get() = withValidityAssertion { backingPsi?.isExpectDeclaration() ?: firSymbol.isExpect }

    override val typeParameters: List<KaTypeParameterSymbol>
        get() = withValidityAssertion {
            with(analysisSession) {
                backingPsi?.getContainingClassOrObject()?.symbol?.typeParameters ?: firSymbol.createKtTypeParameters(builder)
            }
        }

    override fun createPointer(): KaSymbolPointer<KaConstructorSymbol> = withValidityAssertion {
        psiBasedSymbolPointerOfTypeIfSource<KaConstructorSymbol>()?.let { return it }

        KaFirConstructorSymbolPointer(
            analysisSession.createOwnerPointer(this),
            isPrimary,
            FirCallableSignature.createSignature(firSymbol),
        )
    }

    override fun equals(other: Any?): Boolean = psiOrSymbolEquals(other)
    override fun hashCode(): Int = psiOrSymbolHashCode()
}
