/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.fir.annotations.KaFirAnnotationListForDeclaration
import ksp.org.jetbrains.kotlin.analysis.api.fir.hasAnnotation
import ksp.org.jetbrains.kotlin.analysis.api.fir.kaSymbolModalityByModifiers
import ksp.org.jetbrains.kotlin.analysis.api.fir.visibilityByModifiers
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.annotations.KaBaseEmptyAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.symbols.asKaSymbolModality
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.symbols.pointers.KaBasePropertyGetterSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaPropertyGetterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaReceiverParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSymbolModality
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSymbolOrigin
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.types.KaType
import ksp.org.jetbrains.kotlin.descriptors.Visibility
import ksp.org.jetbrains.kotlin.descriptors.annotations.AnnotationUseSiteTarget
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isExpect
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isInline
import ksp.org.jetbrains.kotlin.fir.declarations.utils.modality
import ksp.org.jetbrains.kotlin.fir.declarations.utils.visibility
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirPropertyAccessorSymbol
import ksp.org.jetbrains.kotlin.fir.utils.exceptions.withFirSymbolEntry
import ksp.org.jetbrains.kotlin.lexer.KtTokens
import ksp.org.jetbrains.kotlin.name.CallableId
import ksp.org.jetbrains.kotlin.psi.KtProperty
import ksp.org.jetbrains.kotlin.psi.KtPropertyAccessor
import ksp.org.jetbrains.kotlin.psi.psiUtil.isExpectDeclaration
import ksp.org.jetbrains.kotlin.utils.exceptions.errorWithAttachment
import ksp.org.jetbrains.kotlin.utils.exceptions.requireWithAttachment

/**
 * Represents the default property getter for [KaFirKotlinPropertyKtParameterBasedSymbol] and [KaFirKotlinPropertyKtDestructuringDeclarationEntryBasedSymbol].
 *
 * Also represents [KaFirKotlinPropertyKtPropertyBasedSymbol] with backing [KtPropertyAccessor] without a body.
 */
internal class KaFirDefaultPropertyGetterSymbol(
    val owningKaProperty: KaFirKotlinPropertySymbol<*>,
) : KaPropertyGetterSymbol(), KaFirSymbol<FirPropertyAccessorSymbol> {
    private val backingPsiGetterWithoutBody: KtPropertyAccessor?
        get() = (owningKaProperty.backingPsi as? KtProperty)?.getter

    init {
        requireWithAttachment(
            backingPsiGetterWithoutBody?.hasBody() != true,
            { "This implementation should not be created for property accessor with a body" },
        ) {
            withFirSymbolEntry("property", owningKaProperty.firSymbol)
        }
    }

    override val analysisSession: KaFirSession
        get() = owningKaProperty.analysisSession

    override val firSymbol: FirPropertyAccessorSymbol
        get() = owningKaProperty.firSymbol.getterSymbol ?: errorWithAttachment("Getter is not found") {
            withFirSymbolEntry("property", owningKaProperty.firSymbol)
        }

    override val psi: PsiElement?
        get() = withValidityAssertion { backingPsiGetterWithoutBody }

    override val isExpect: Boolean
        get() = withValidityAssertion {
            backingPsiGetterWithoutBody?.hasModifier(KtTokens.EXPECT_KEYWORD) == true ||
                    owningKaProperty.backingPsi?.isExpectDeclaration() ?: firSymbol.isExpect
        }

    override val isDefault: Boolean
        get() = withValidityAssertion { true }

    override val isInline: Boolean
        get() = withValidityAssertion {
            backingPsiGetterWithoutBody?.hasModifier(KtTokens.INLINE_KEYWORD) == true ||
                    owningKaProperty.backingPsi?.hasModifier(KtTokens.INLINE_KEYWORD) ?: firSymbol.isInline
        }

    override val isOverride: Boolean
        get() = withValidityAssertion { owningKaProperty.isOverride }

    override val hasBody: Boolean
        get() = withValidityAssertion { false }

    override val modality: KaSymbolModality
        get() = withValidityAssertion {
            backingPsiGetterWithoutBody?.kaSymbolModalityByModifiers
                ?: owningKaProperty.modalityByPsi
                ?: firSymbol.modality.asKaSymbolModality
        }

    override val compilerVisibility: Visibility
        get() = withValidityAssertion {
            backingPsiGetterWithoutBody?.visibilityByModifiers
                ?: owningKaProperty.compilerVisibilityByPsi
                ?: firSymbol.visibility
        }

    override val returnType: KaType
        get() = withValidityAssertion { firSymbol.returnType(builder) }

    override val receiverParameter: KaReceiverParameterSymbol?
        get() = withValidityAssertion {
            owningKaProperty.receiverParameter
        }

    override val annotations: KaAnnotationList
        get() = withValidityAssertion {
            val backingPsi = owningKaProperty.backingPsi
            if (backingPsi != null &&
                !backingPsi.hasAnnotation(AnnotationUseSiteTarget.PROPERTY_GETTER) &&
                backingPsiGetterWithoutBody?.annotationEntries.isNullOrEmpty()
            )
                KaBaseEmptyAnnotationList(token)
            else
                KaFirAnnotationListForDeclaration.create(firSymbol, builder)
        }

    override val callableId: CallableId?
        get() = withValidityAssertion { null }

    override val origin: KaSymbolOrigin
        get() = withValidityAssertion { owningKaProperty.origin }

    override fun createPointer(): KaSymbolPointer<KaPropertyGetterSymbol> = withValidityAssertion {
        KaBasePropertyGetterSymbolPointer(owningKaProperty.createPointer())
    }

    override fun equals(other: Any?): Boolean = this === other ||
            other is KaFirDefaultPropertyGetterSymbol &&
            other.owningKaProperty == owningKaProperty

    override fun hashCode(): Int = 31 * owningKaProperty.hashCode() + KaFirKotlinPropertySymbol.HASH_CODE_ADDITION_FOR_GETTER
}
