/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols

import ksp.com.intellij.openapi.project.Project
import ksp.com.intellij.psi.JavaPsiFacade
import ksp.com.intellij.psi.PsiElement
import ksp.com.intellij.psi.PsiManager
import ksp.com.intellij.psi.impl.file.PsiPackageImpl
import ksp.com.intellij.psi.search.GlobalSearchScope
import ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.pointers.KaFirPackageSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.fir.utils.cached
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.KaLifetimeOwner
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.KaLifetimeToken
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.platform.packages.createPackageProvider
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaPackageSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSymbolOrigin
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.name.FqName

internal class KaFirPackageSymbol(
    override val fqName: FqName,
    private val project: Project,
    override val token: KaLifetimeToken,
) : KaPackageSymbol(), KaLifetimeOwner {
    override val psi: PsiElement? by cached {
        JavaPsiFacade.getInstance(project).findPackage(fqName.asString())
            ?: KtPackage(PsiManager.getInstance(project), fqName, GlobalSearchScope.allScope(project)/*TODO*/)
    }

    override val origin: KaSymbolOrigin
        get() = withValidityAssertion { KaSymbolOrigin.SOURCE } // TODO

    override fun createPointer(): KaSymbolPointer<KaPackageSymbol> = withValidityAssertion {
        KaFirPackageSymbolPointer(fqName)
    }

    override fun equals(other: Any?): Boolean = this === other || other is KaFirPackageSymbol && other.fqName == fqName
    override fun hashCode(): Int = fqName.hashCode()
}

private class KtPackage(
    manager: PsiManager,
    private val fqName: FqName,
    private val scope: GlobalSearchScope,
) : PsiPackageImpl(manager, fqName.asString().replace('/', '.')) {
    override fun copy() = KtPackage(manager, fqName, scope)
    override fun isValid(): Boolean = project.createPackageProvider(scope).doesKotlinOnlyPackageExist(fqName)
}
