/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.fir.annotations.KaFirAnnotationListForDeclaration
import ksp.org.jetbrains.kotlin.analysis.api.fir.findPsi
import ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.pointers.createOwnerPointer
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.symbols.pointers.KaBasePropertySetterSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaPropertySetterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaReceiverParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSymbolModality
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaValueParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.types.KaType
import ksp.org.jetbrains.kotlin.descriptors.Visibility
import ksp.org.jetbrains.kotlin.fir.declarations.utils.hasBody
import ksp.org.jetbrains.kotlin.fir.declarations.utils.hasStableParameterNames
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isOverride
import ksp.org.jetbrains.kotlin.fir.declarations.utils.visibility
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirSyntheticPropertyAccessorSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.isExtension
import ksp.org.jetbrains.kotlin.name.CallableId

internal class KaFirSyntheticPropertySetterSymbol(
    override val firSymbol: FirSyntheticPropertyAccessorSymbol,
    override val analysisSession: KaFirSession,
) : KaPropertySetterSymbol(), KaFirSymbol<FirSyntheticPropertyAccessorSymbol> {
    init {
        require(firSymbol.isSetter)
    }

    override val psi: PsiElement?
        get() = withValidityAssertion { firSymbol.findPsi() }

    override val isDefault: Boolean
        get() = withValidityAssertion { false }

    override val isInline: Boolean
        get() = withValidityAssertion { false }

    override val isOverride: Boolean
        get() = withValidityAssertion {
            firSymbol.isOverride || firSymbol.fir.propertySymbol.isOverride
        }

    override val isExtension: Boolean
        get() = withValidityAssertion { firSymbol.isExtension }

    override val hasBody: Boolean
        get() = withValidityAssertion { firSymbol.fir.hasBody }

    override val modality: KaSymbolModality
        get() = withValidityAssertion { firSymbol.kaSymbolModality }

    override val compilerVisibility: Visibility
        get() = withValidityAssertion { firSymbol.visibility }

    override val returnType: KaType
        get() = withValidityAssertion { firSymbol.returnType(builder) }

    override val receiverParameter: KaReceiverParameterSymbol?
        get() = withValidityAssertion { KaFirReceiverParameterSymbol.create(null, analysisSession, this) }

    override val annotations: KaAnnotationList
        get() = withValidityAssertion {
            KaFirAnnotationListForDeclaration.create(firSymbol, builder)
        }

    override val callableId: CallableId?
        get() = withValidityAssertion { firSymbol.delegateFunctionSymbol.callableId }

    override val parameter: KaValueParameterSymbol
        get() = withValidityAssertion { firSymbol.createKtValueParameters(builder).single() }

    override val hasStableParameterNames: Boolean
        get() = withValidityAssertion { firSymbol.fir.hasStableParameterNames }

    override fun createPointer(): KaSymbolPointer<KaPropertySetterSymbol> = withValidityAssertion {
        KaBasePropertySetterSymbolPointer(propertySymbolPointer = analysisSession.createOwnerPointer(this))
    }

    override fun equals(other: Any?): Boolean = symbolEquals(other)
    override fun hashCode(): Int = symbolHashCode()
}