/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.pointers

import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSyntheticJavaPropertySymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.markers.KaDeclarationContainerSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.synthetic.FirSyntheticProperty
import ksp.org.jetbrains.kotlin.fir.resolve.calls.FirSyntheticPropertiesScope
import ksp.org.jetbrains.kotlin.fir.resolve.defaultType
import ksp.org.jetbrains.kotlin.fir.scopes.FirScope
import ksp.org.jetbrains.kotlin.fir.scopes.FirTypeScope
import ksp.org.jetbrains.kotlin.fir.scopes.getProperties
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirClassSymbol
import ksp.org.jetbrains.kotlin.name.Name

internal class KaFirJavaSyntheticPropertySymbolPointer(
    ownerPointer: KaSymbolPointer<KaDeclarationContainerSymbol>,
    private val propertyName: Name,
    private val isSynthetic: Boolean,
) : KaFirMemberSymbolPointer<KaSyntheticJavaPropertySymbol>(ownerPointer) {
    override fun KaFirSession.chooseCandidateAndCreateSymbol(
        candidates: FirScope,
        firSession: FirSession,
    ): KaSyntheticJavaPropertySymbol? {
        val syntheticProperty = candidates.getProperties(propertyName)
            .mapNotNull { it.fir as? FirSyntheticProperty }
            .singleOrNull()
            ?: return null

        return firSymbolBuilder.variableBuilder.buildSyntheticJavaPropertySymbol(syntheticProperty.symbol)
    }

    override fun getSearchScope(analysisSession: KaFirSession, owner: FirClassSymbol<*>): FirScope? {
        val baseScope = super.getSearchScope(analysisSession, owner) as? FirTypeScope ?: return null
        return if (isSynthetic) {
            FirSyntheticPropertiesScope.createIfSyntheticNamesProviderIsDefined(
                session = analysisSession.firSession,
                dispatchReceiverType = owner.defaultType(),
                baseScope = baseScope,
            )
        } else {
            baseScope
        }
    }

    override fun pointsToTheSameSymbolAs(other: KaSymbolPointer<KaSymbol>): Boolean = this === other ||
            other is KaFirJavaSyntheticPropertySymbolPointer &&
            other.propertyName == propertyName &&
            other.isSynthetic == isSynthetic &&
            hasTheSameOwner(other)
}
