/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.util

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind.*
import ksp.org.jetbrains.kotlin.KtPsiSourceElement
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.api.getFirResolveSession
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.api.resolveToFirSymbolOfTypeSafe
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.projectStructure.llFirModuleData
import ksp.org.jetbrains.kotlin.fir.declarations.isLazyResolvable
import ksp.org.jetbrains.kotlin.fir.symbols.FirBasedSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.*
import ksp.org.jetbrains.kotlin.psi.KtCallableDeclaration
import ksp.org.jetbrains.kotlin.psi.KtClassOrObject
import ksp.org.jetbrains.kotlin.psi.KtDeclaration
import ksp.org.jetbrains.kotlin.psi.KtEnumEntry
import ksp.org.jetbrains.kotlin.psi.KtParameter
import ksp.org.jetbrains.kotlin.psi.KtProperty
import ksp.org.jetbrains.kotlin.psi.KtPropertyAccessor
import ksp.org.jetbrains.kotlin.psi.psiUtil.containingClassOrObject

internal object LLContainingClassCalculator {
    /**
     * Returns a containing class symbol for the given symbol, computing it solely from the source information
     * and information inside FIR nodes.
     */
    fun getContainingClassSymbol(symbol: FirBasedSymbol<*>): FirClassLikeSymbol<*>? {
        if (!symbol.origin.isLazyResolvable) {
            // Handle only source or source-based declarations for now as below we use the PSI tree
            return null
        }

        if (symbol is FirAnonymousInitializerSymbol) {
            // For anonymous initializers, the containing class symbol is right there, no need in PSI traversal
            return symbol.containingDeclarationSymbol as? FirClassLikeSymbol<*>
        }

        if (!canHaveContainingClassSymbol(symbol)) {
            return null
        }

        val source = symbol.source as? KtPsiSourceElement ?: return null
        val kind = source.kind

        when (kind) {
            is KtFakeSourceElementKind -> {
                if (symbol is FirConstructorSymbol && kind == ImplicitConstructor) {
                    return computeContainingClass(symbol, source.psi)
                }

                if (symbol is FirPropertyAccessorSymbol) {
                    if (kind == DefaultAccessor) {
                        val containingProperty = source.psi
                        return if (containingProperty is KtProperty || containingProperty is KtParameter) {
                            computeContainingClass(symbol, (containingProperty as KtDeclaration).containingClassOrObject)
                        } else {
                            null
                        }
                    }

                    if (kind == DelegatedPropertyAccessor) {
                        val containingProperty = source.psi as? KtProperty
                        return computeContainingClass(symbol, containingProperty?.containingClassOrObject)
                    }

                    if (kind == PropertyFromParameter) {
                        val containingParameter = source.psi as? KtParameter
                        return computeContainingClass(symbol, containingParameter?.containingClassOrObject)
                    }
                }

                if (symbol is FirPropertySymbol && kind == PropertyFromParameter) {
                    val containingParameter = source.psi as? KtParameter
                    return computeContainingClass(symbol, containingParameter?.containingClassOrObject)
                }

                if (kind == EnumGeneratedDeclaration) {
                    return computeContainingClass(symbol, source.psi)
                }

                if (kind == DataClassGeneratedMembers) {
                    val containingClass = when (val psi = source.psi) {
                        is KtClassOrObject -> psi
                        is KtParameter -> psi.containingClassOrObject // component() functions point to 'KtParameter's
                        else -> null
                    }
                    return computeContainingClass(symbol, containingClass)
                }
            }
            else -> {
                if (symbol is FirClassLikeSymbol<*>) {
                    val selfClass = source.psi as? KtClassOrObject
                    return computeContainingClass(symbol, selfClass?.containingClassOrObject)
                }

                if (symbol is FirCallableSymbol<*>) {
                    val selfCallable = source.psi
                    return when (selfCallable) {
                        is KtCallableDeclaration, is KtEnumEntry -> {
                            computeContainingClass(symbol, selfCallable.containingClassOrObject)
                        }
                        is KtPropertyAccessor -> {
                            val containingProperty = selfCallable.property
                            computeContainingClass(symbol, containingProperty.containingClassOrObject)
                        }
                        else -> null
                    }
                }
            }
        }

        return null
    }

    private fun canHaveContainingClassSymbol(symbol: FirBasedSymbol<*>): Boolean {
        return when (symbol) {
            is FirValueParameterSymbol, is FirAnonymousFunctionSymbol -> false
            is FirPropertySymbol -> !symbol.isLocal
            is FirCallableSymbol, is FirClassLikeSymbol -> true
            is FirDanglingModifierSymbol -> true
            else -> false
        }
    }

    private fun computeContainingClass(symbol: FirBasedSymbol<*>, psi: PsiElement?): FirClassLikeSymbol<*>? {
        if (psi !is KtClassOrObject) {
            return null
        }

        val module = symbol.llFirModuleData.ktModule
        val resolveSession = module.getFirResolveSession(module.project)
        return psi.resolveToFirSymbolOfTypeSafe<FirClassLikeSymbol<*>>(resolveSession)
    }
}