/*
 * Copyright 2010-2019 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.backend.jvm.lower

import ksp.org.jetbrains.kotlin.backend.common.lower.InventNamesForLocalClasses
import ksp.org.jetbrains.kotlin.backend.common.phaser.PhaseDescription
import ksp.org.jetbrains.kotlin.backend.jvm.JvmBackendContext
import ksp.org.jetbrains.kotlin.backend.jvm.localClassType
import ksp.org.jetbrains.kotlin.codegen.sanitizeNameIfNeeded
import ksp.org.jetbrains.kotlin.ir.declarations.IrAttributeContainer
import ksp.org.jetbrains.kotlin.ir.declarations.IrClass
import ksp.org.jetbrains.kotlin.ir.declarations.IrDeclarationOrigin
import ksp.org.jetbrains.kotlin.ir.declarations.IrFile
import ksp.org.jetbrains.kotlin.ir.util.render
import ksp.org.jetbrains.kotlin.resolve.jvm.JvmClassName
import ksp.org.jetbrains.org.objectweb.asm.Type

@PhaseDescription(
    name = "InventNamesForLocalClasses",
    // MainMethodGeneration introduces lambdas, needing names for their local classes.
    prerequisite = [MainMethodGenerationLowering::class],
)
internal class JvmInventNamesForLocalClasses(private val context: JvmBackendContext) : InventNamesForLocalClasses() {
    override fun computeTopLevelClassName(clazz: IrClass): String {
        val file = clazz.parent as? IrFile
            ?: throw AssertionError("Top-level class expected: ${clazz.render()}")
        val classFqn =
            if (clazz.origin == IrDeclarationOrigin.FILE_CLASS ||
                clazz.origin == IrDeclarationOrigin.SYNTHETIC_FILE_CLASS
            ) {
                file.getFileClassInfo().fileClassFqName
            } else {
                file.packageFqName.child(clazz.name)
            }
        return JvmClassName.byFqNameWithoutInnerClasses(classFqn).internalName
    }

    override fun sanitizeNameIfNeeded(name: String): String {
        return sanitizeNameIfNeeded(name, context.config.languageVersionSettings)
    }

    override fun putLocalClassName(declaration: IrAttributeContainer, localClassName: String) {
        // We can visit the same class twice: before IR inlining and after. The name that was before is more preferable.
        if (declaration.localClassType != null) return
        declaration.localClassType = Type.getObjectType(localClassName)
    }
}
