/*
 * Copyright 2010-2019 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.backend.jvm.lower

import ksp.org.jetbrains.kotlin.backend.common.FileLoweringPass
import ksp.org.jetbrains.kotlin.backend.common.phaser.PhaseDescription
import ksp.org.jetbrains.kotlin.backend.jvm.JvmBackendContext
import ksp.org.jetbrains.kotlin.ir.IrElement
import ksp.org.jetbrains.kotlin.ir.declarations.IrDeclarationWithName
import ksp.org.jetbrains.kotlin.ir.declarations.IrFile
import ksp.org.jetbrains.kotlin.ir.expressions.IrLoop
import ksp.org.jetbrains.kotlin.ir.visitors.IrElementVisitorVoid
import ksp.org.jetbrains.kotlin.ir.visitors.acceptChildrenVoid
import ksp.org.jetbrains.kotlin.ir.visitors.acceptVoid
import ksp.org.jetbrains.kotlin.name.Name

/**
 * Labels all loops for non-local `break`/`continue`.
 */
@PhaseDescription(name = "UniqueLoopLabels")
internal class UniqueLoopLabelsLowering(@Suppress("UNUSED_PARAMETER", "unused") context: JvmBackendContext) : FileLoweringPass {
    override fun lower(irFile: IrFile) {
        irFile.acceptVoid(object : IrElementVisitorVoid {
            // This counter is intentionally not local to every declaration because their names might clash.
            private var counter = 0
            private val stack = ArrayList<Name>()

            override fun visitElement(element: IrElement) {
                if (element is IrDeclarationWithName) {
                    stack.add(element.name)
                    element.acceptChildrenVoid(this)
                    stack.removeLast()
                } else {
                    element.acceptChildrenVoid(this)
                }
            }

            override fun visitLoop(loop: IrLoop) {
                // Give all loops unique labels so that we can generate unambiguous instructions for non-local
                // `break`/`continue` statements.
                loop.label = stack.joinToString("$", postfix = (++counter).toString())
                super.visitLoop(loop)
            }
        })
    }
}
