/*
 * Copyright 2010-2022 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.backend.wasm.ir2wasm

import ksp.org.jetbrains.kotlin.ir.IrElement
import ksp.org.jetbrains.kotlin.ir.IrFileEntry
import ksp.org.jetbrains.kotlin.ir.LineAndColumn
import ksp.org.jetbrains.kotlin.ir.UNDEFINED_OFFSET
import ksp.org.jetbrains.kotlin.ir.declarations.IrFile
import ksp.org.jetbrains.kotlin.ir.declarations.IrSymbolOwner
import ksp.org.jetbrains.kotlin.ir.symbols.IrFunctionSymbol
import ksp.org.jetbrains.kotlin.ir.symbols.IrSymbol
import ksp.org.jetbrains.kotlin.ir.util.SYNTHETIC_OFFSET
import ksp.org.jetbrains.kotlin.ir.util.getPackageFragment
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.wasm.ir.WasmExpressionBuilder
import ksp.org.jetbrains.kotlin.wasm.ir.source.location.SourceLocation

private val IrElement.hasSyntheticOrUndefinedLocation: Boolean
    get() = startOffset in SYNTHETIC_OFFSET..UNDEFINED_OFFSET ||
            endOffset in SYNTHETIC_OFFSET..UNDEFINED_OFFSET

enum class LocationType {
    START {
        override fun getLineAndColumnNumberFor(irElement: IrElement, fileEntry: IrFileEntry) =
            fileEntry.getLineAndColumnNumbers(irElement.startOffset)
    },
    END {
        override fun getLineAndColumnNumberFor(irElement: IrElement, fileEntry: IrFileEntry) =
            fileEntry.getLineAndColumnNumbers(irElement.endOffset)
    };

    abstract fun getLineAndColumnNumberFor(irElement: IrElement, fileEntry: IrFileEntry): LineAndColumn
}

private val IrSymbol?.shouldIgnore: Boolean
    get() = this?.owner?.getPackageFragment()?.packageFqName?.startsWith(StandardClassIds.BASE_KOTLIN_PACKAGE) == true

fun IrElement.getSourceLocation(
    declaration: IrSymbol?,
    file: IrFile?,
    type: LocationType = LocationType.START
): SourceLocation {
    val fileEntry = file?.fileEntry

    if (fileEntry == null) return SourceLocation.NoLocation("fileEntry is null")
    if (hasSyntheticOrUndefinedLocation) return SourceLocation.NoLocation("Synthetic declaration")

    val path = fileEntry.name
    val (line, column) = type.getLineAndColumnNumberFor(this, fileEntry)

    if (line < 0 || column < 0) return SourceLocation.NoLocation("startLine or startColumn < 0")

    // TODO Drop "file" usages after KT-58406 fix and replace IrFile with IrFileEntry
    val module = file.module.name.asString()

    return if (declaration.shouldIgnore) {
        SourceLocation.IgnoredLocation(
            module,
            path,
            line,
            column
        )
    } else {
        SourceLocation.Location(
            module,
            path,
            line,
            column
        )
    }
}

fun WasmExpressionBuilder.buildUnreachableForVerifier() {
    buildUnreachable(SourceLocation.NoLocation("This instruction should never be reached, but required for wasm verifier"))
}

fun WasmExpressionBuilder.buildUnreachableAfterNothingType() {
    buildUnreachable(
        SourceLocation.NoLocation(
            "The unreachable instruction after an expression with Nothing type to make sure that " +
                    "execution doesn't come here (or it fails fast if so). It also might be required for wasm verifier."
        )
    )
}
