/*
 * Copyright 2000-2018 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.codegen

import ksp.org.jetbrains.kotlin.builtins.KotlinBuiltIns
import ksp.org.jetbrains.kotlin.codegen.state.KotlinTypeMapper
import ksp.org.jetbrains.kotlin.config.LanguageVersionSettings
import ksp.org.jetbrains.kotlin.descriptors.DeclarationDescriptor
import ksp.org.jetbrains.kotlin.psi.KtExpression
import ksp.org.jetbrains.kotlin.resolve.BindingContext
import ksp.org.jetbrains.kotlin.resolve.calls.smartcasts.DataFlowValueFactoryImpl
import ksp.org.jetbrains.kotlin.resolve.calls.smartcasts.getKotlinTypeForComparison
import ksp.org.jetbrains.kotlin.resolve.calls.smartcasts.getKotlinTypeWithPossibleSmartCastToFP
import ksp.org.jetbrains.kotlin.types.KotlinType
import ksp.org.jetbrains.kotlin.types.TypeUtils
import ksp.org.jetbrains.org.objectweb.asm.Type

class TypeAndNullability(@JvmField val type: Type, @JvmField val isNullable: Boolean)

fun calcProperTypeForIeee754ArithmeticIfNeeded(
    expression: KtExpression,
    bindingContext: BindingContext,
    inferredPrimitiveType: KotlinType?,
    typeMapper: KotlinTypeMapper
): TypeAndNullability? {
    if (inferredPrimitiveType == null) return null
    val ktType = expression.getKotlinTypeForComparison(bindingContext) ?: return null
    val isNullable = TypeUtils.isNullableType(ktType)
    val asmType = typeMapper.mapType(inferredPrimitiveType)
    if (!AsmUtil.isPrimitive(asmType)) return null
    return TypeAndNullability(asmType, isNullable)
}

fun legacyCalcTypeForIeee754ArithmeticIfNeeded(
    expression: KtExpression?,
    bindingContext: BindingContext,
    descriptor: DeclarationDescriptor,
    languageVersionSettings: LanguageVersionSettings
): TypeAndNullability? {
    val ktType = expression.getKotlinTypeWithPossibleSmartCastToFP(
        // NB. Using DataFlowValueFactoryImpl is a hack, but it is ok for 'legacy'
        bindingContext, descriptor, languageVersionSettings, DataFlowValueFactoryImpl(languageVersionSettings)
    )

    if (ktType != null) {
        if (KotlinBuiltIns.isDoubleOrNullableDouble(ktType)) {
            return TypeAndNullability(
                Type.DOUBLE_TYPE,
                TypeUtils.isNullableType(ktType)
            )
        }

        if (KotlinBuiltIns.isFloatOrNullableFloat(ktType)) {
            return TypeAndNullability(
                Type.FLOAT_TYPE,
                TypeUtils.isNullableType(ktType)
            )
        }
    }
    return null
}