/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclarationOrigin
import ksp.org.jetbrains.kotlin.fir.declarations.FirRegularClass
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnnotation
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import ksp.org.jetbrains.kotlin.fir.references.toResolvedConstructorSymbol
import ksp.org.jetbrains.kotlin.fir.types.ConeCapturedType
import ksp.org.jetbrains.kotlin.fir.types.coneType
import ksp.org.jetbrains.kotlin.fir.types.contains
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol

object FirConstructorCallChecker : FirFunctionCallChecker(MppCheckerKind.Common) {
    override fun check(expression: FirFunctionCall, context: CheckerContext, reporter: DiagnosticReporter) {
        val constructorSymbol = expression.calleeReference.toResolvedConstructorSymbol() ?: return
        val coneType = constructorSymbol.resolvedReturnTypeRef.coneType

        if (constructorSymbol.origin == FirDeclarationOrigin.Synthetic.TypeAliasConstructor && coneType.contains { it is ConeCapturedType }) {
            reporter.reportOn(expression.source, FirErrors.CONSTRUCTOR_OR_SUPERTYPE_ON_TYPEALIAS_WITH_TYPE_PROJECTION, context)
        }

        val declarationClass = coneType.toRegularClassSymbol(context.session)
            ?: return

        if (declarationClass.classKind == ClassKind.ANNOTATION_CLASS &&
            context.callsOrAssignments.all { call ->
                call !is FirAnnotation
            } &&
            context.containingDeclarations.all { klass ->
                klass !is FirRegularClass || klass.classKind != ClassKind.ANNOTATION_CLASS
            }
        ) {
            if (!context.languageVersionSettings.supportsFeature(LanguageFeature.InstantiationOfAnnotationClasses)) reporter.reportOn(
                expression.source,
                FirErrors.ANNOTATION_CLASS_CONSTRUCTOR_CALL,
                context
            )
        }

        if (declarationClass.classKind == ClassKind.ENUM_CLASS) {
            reporter.reportOn(
                expression.source,
                FirErrors.ENUM_CLASS_CONSTRUCTOR_CALL,
                context
            )
        }
    }
}
