/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.expressions.FirLiteralExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import ksp.org.jetbrains.kotlin.fir.expressions.arguments
import ksp.org.jetbrains.kotlin.fir.references.toResolvedNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.name.Name

object FirDivisionByZeroChecker : FirFunctionCallChecker(MppCheckerKind.Common) {
    private val defaultPackageName = FqName("kotlin")
    private val defaultDivName = Name.identifier("div")

    override fun check(expression: FirFunctionCall, context: CheckerContext, reporter: DiagnosticReporter) {
        val firstValue = (expression.arguments.singleOrNull() as? FirLiteralExpression)?.value
        if (firstValue != null && (firstValue == 0L || firstValue == 0.0f || firstValue == 0.0)) {
            val callableId = (expression.calleeReference.toResolvedNamedFunctionSymbol())?.callableId
            if (callableId != null && callableId.packageName == defaultPackageName && callableId.callableName == defaultDivName) {
                reporter.reportOn(expression.source, FirErrors.DIVISION_BY_ZERO, context)
            }
        }
    }
}
