/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.KtDiagnosticFactory0
import ksp.org.jetbrains.kotlin.diagnostics.KtDiagnosticFactory1
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.valOrVarKeyword
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.HAS_NEXT_FUNCTION_AMBIGUITY
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.HAS_NEXT_FUNCTION_NONE_APPLICABLE
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.HAS_NEXT_MISSING
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.ITERATOR_AMBIGUITY
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.ITERATOR_MISSING
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.ITERATOR_ON_NULLABLE
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.NEXT_AMBIGUITY
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.NEXT_MISSING
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.NEXT_NONE_APPLICABLE
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.OPERATOR_MODIFIER_REQUIRED
import ksp.org.jetbrains.kotlin.fir.declarations.FirProperty
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isOperator
import ksp.org.jetbrains.kotlin.fir.expressions.FirBlock
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import ksp.org.jetbrains.kotlin.fir.expressions.FirWhileLoop
import ksp.org.jetbrains.kotlin.fir.references.FirResolvedNamedReference
import ksp.org.jetbrains.kotlin.fir.references.isError
import ksp.org.jetbrains.kotlin.fir.resolve.calls.OperatorCallOfNonOperatorFunction
import ksp.org.jetbrains.kotlin.fir.resolve.calls.InapplicableNullableReceiver
import ksp.org.jetbrains.kotlin.fir.resolve.diagnostics.*
import ksp.org.jetbrains.kotlin.fir.symbols.FirBasedSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.util.OperatorNameConventions

object FirForLoopChecker : FirBlockChecker(MppCheckerKind.Common) {
    override fun check(expression: FirBlock, context: CheckerContext, reporter: DiagnosticReporter) {
        if (expression.source?.kind != KtFakeSourceElementKind.DesugaredForLoop) return

        val statements = expression.statements
        val iteratorDeclaration = statements[0] as? FirProperty ?: return
        val whileLoop = statements[1] as? FirWhileLoop ?: return
        if (iteratorDeclaration.source?.kind != KtFakeSourceElementKind.DesugaredForLoop) return
        val iteratorCall = iteratorDeclaration.initializer as FirFunctionCall
        val source = iteratorCall.explicitReceiver?.source ?: iteratorCall.source
        if (checkSpecialFunctionCall(
                iteratorCall,
                reporter,
                source,
                context,
                ITERATOR_AMBIGUITY,
                ITERATOR_MISSING,
                nullableReceiverFactory = ITERATOR_ON_NULLABLE
            )
        ) {
            return
        }

        val hasNextCall = whileLoop.condition as FirFunctionCall
        checkSpecialFunctionCall(
            hasNextCall,
            reporter,
            source,
            context,
            HAS_NEXT_FUNCTION_AMBIGUITY,
            HAS_NEXT_MISSING,
            noneApplicableFactory = HAS_NEXT_FUNCTION_NONE_APPLICABLE
        )

        val loopParameter = whileLoop.block.statements.firstOrNull() as? FirProperty ?: return
        if (loopParameter.initializer?.source?.kind != KtFakeSourceElementKind.DesugaredForLoop) return
        val nextCall = loopParameter.initializer as FirFunctionCall
        checkSpecialFunctionCall(
            nextCall,
            reporter,
            source,
            context,
            NEXT_AMBIGUITY,
            NEXT_MISSING,
            noneApplicableFactory = NEXT_NONE_APPLICABLE
        )

        val loopParameterSource = loopParameter.source
        loopParameterSource.valOrVarKeyword?.let {
            reporter.reportOn(loopParameterSource, FirErrors.VAL_OR_VAR_ON_LOOP_PARAMETER, it, context)
        }
    }

    private fun checkSpecialFunctionCall(
        call: FirFunctionCall,
        reporter: DiagnosticReporter,
        reportSource: KtSourceElement?,
        context: CheckerContext,
        ambiguityFactory: KtDiagnosticFactory1<Collection<FirBasedSymbol<*>>>,
        missingFactory: KtDiagnosticFactory0,
        noneApplicableFactory: KtDiagnosticFactory1<Collection<FirBasedSymbol<*>>>? = null,
        nullableReceiverFactory: KtDiagnosticFactory0? = null,
    ): Boolean {
        val calleeReference = call.calleeReference
        when {
            calleeReference.isError() -> {
                when (val diagnostic = calleeReference.diagnostic) {
                    is ConeAmbiguityError -> {
                        reporter.reportOn(
                            reportSource,
                            noneApplicableFactory ?: ambiguityFactory,
                            diagnostic.candidates.map { it.symbol },
                            context
                        )
                    }
                    is ConeUnresolvedNameError -> {
                        reporter.reportOn(reportSource, missingFactory, context)
                    }
                    is ConeInapplicableWrongReceiver -> when {
                        noneApplicableFactory != null -> {
                            reporter.reportOn(reportSource, noneApplicableFactory, diagnostic.candidateSymbols, context)
                        }
                        calleeReference.name == OperatorNameConventions.ITERATOR -> {
                            reporter.reportOn(reportSource, missingFactory, context)
                        }
                        else -> {
                            error("ConeInapplicableWrongReceiver, but no diagnostic reported")
                        }
                    }
                    is ConeConstraintSystemHasContradiction -> {
                        if (calleeReference.name == OperatorNameConventions.ITERATOR)
                            reporter.reportOn(reportSource, missingFactory, context)
                    }
                    is ConeInapplicableCandidateError -> {
                        if (nullableReceiverFactory != null || noneApplicableFactory != null) {
                            diagnostic.candidate.diagnostics.filter { it.applicability == diagnostic.applicability }.forEach {
                                when (it) {
                                    is InapplicableNullableReceiver -> {
                                        if (nullableReceiverFactory != null) {
                                            reporter.reportOn(
                                                reportSource, nullableReceiverFactory, context
                                            )
                                        } else {
                                            reporter.reportOn(
                                                reportSource, noneApplicableFactory!!, listOf(diagnostic.candidate.symbol), context
                                            )
                                        }
                                        return true
                                    }
                                    is OperatorCallOfNonOperatorFunction -> {
                                        val symbol = it.function
                                        reporter.reportOn(
                                            reportSource, OPERATOR_MODIFIER_REQUIRED, symbol, symbol.name.asString(), context
                                        )
                                    }
                                }
                            }
                        }
                    }
                }
                return true
            }
            calleeReference is FirResolvedNamedReference -> {
                val symbol = calleeReference.resolvedSymbol
                if (symbol is FirNamedFunctionSymbol) {
                    if (!symbol.isOperator) {
                        reporter.reportOn(reportSource, OPERATOR_MODIFIER_REQUIRED, symbol, symbol.name.asString(), context)
                        // Don't return true as we want to report other errors
                    }
                }
            }
        }
        return false
    }
}
