/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.descriptors.SourceElement
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.FirElement
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.utils.sourceElement
import ksp.org.jetbrains.kotlin.fir.expressions.FirQualifiedAccessExpression
import ksp.org.jetbrains.kotlin.fir.references.toResolvedCallableSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.types.ConeKotlinType
import ksp.org.jetbrains.kotlin.fir.types.coneType
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.serialization.deserialization.descriptors.DeserializedContainerAbiStability
import ksp.org.jetbrains.kotlin.serialization.deserialization.descriptors.DeserializedContainerSource

object FirIncompatibleClassExpressionChecker : FirQualifiedAccessExpressionChecker(MppCheckerKind.Common) {
    override fun check(expression: FirQualifiedAccessExpression, context: CheckerContext, reporter: DiagnosticReporter) {
        val symbol = expression.calleeReference.toResolvedCallableSymbol() ?: return

        checkType(symbol.resolvedReturnType, expression, context, reporter)
        checkType(symbol.receiverParameter?.typeRef?.coneType, expression, context, reporter)
        if (symbol is FirFunctionSymbol) {
            for (parameter in symbol.valueParameterSymbols) {
                checkType(parameter.resolvedReturnTypeRef.coneType, expression, context, reporter)
            }
        }

        checkSourceElement(symbol.containerSource, expression, context, reporter)
    }

    internal fun checkType(type: ConeKotlinType?, element: FirElement, context: CheckerContext, reporter: DiagnosticReporter) {
        val classSymbol = type?.toRegularClassSymbol(context.session)
        checkSourceElement(classSymbol?.sourceElement, element, context, reporter)
    }

    private fun checkSourceElement(source: SourceElement?, element: FirElement, context: CheckerContext, reporter: DiagnosticReporter) {
        if (source !is DeserializedContainerSource) return

        val incompatibility = source.incompatibility
        if (incompatibility != null) {
            reporter.reportOn(element.source, FirErrors.INCOMPATIBLE_CLASS, source.presentableString, incompatibility, context)
        }
        if (source.isPreReleaseInvisible) {
            reporter.reportOn(element.source, FirErrors.PRE_RELEASE_CLASS, source.presentableString, context)
        }
        if (source.abiStability == DeserializedContainerAbiStability.UNSTABLE) {
            reporter.reportOn(element.source, FirErrors.IR_WITH_UNSTABLE_ABI_COMPILED_CLASS, source.presentableString, context)
        }
    }
}
