/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.isExplicit
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.isStandalone
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.fullyExpandedClass
import ksp.org.jetbrains.kotlin.fir.expressions.FirResolvedQualifier
import ksp.org.jetbrains.kotlin.fir.symbols.FirBasedSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirTypeAliasSymbol
import ksp.org.jetbrains.kotlin.fir.types.isUnit
import ksp.org.jetbrains.kotlin.fir.types.resolvedType

object FirStandaloneQualifierChecker : FirResolvedQualifierChecker(MppCheckerKind.Common) {
    override fun check(expression: FirResolvedQualifier, context: CheckerContext, reporter: DiagnosticReporter) {
        if (!expression.isStandalone(context)) return

        // Note: if it's real Unit, it will be filtered by ClassKind.OBJECT check below in reportErrorOn
        if (!expression.resolvedType.isUnit) {
            if (expression.typeArguments.any { it.isExplicit }) {
                reporter.reportOn(expression.source, FirErrors.EXPLICIT_TYPE_ARGUMENTS_IN_PROPERTY_ACCESS, "Object", context)
            }
            return
        }

        expression.symbol.reportErrorOn(expression.source, context, reporter)
    }

    private fun FirBasedSymbol<*>?.reportErrorOn(source: KtSourceElement?, context: CheckerContext, reporter: DiagnosticReporter) {
        when (this) {
            is FirRegularClassSymbol -> {
                if (classKind == ClassKind.OBJECT) return
                reporter.reportOn(source, FirErrors.NO_COMPANION_OBJECT, this, context)
            }
            is FirTypeAliasSymbol -> {
                fullyExpandedClass(context.session)?.reportErrorOn(source, context, reporter)
            }
            null -> {
                reporter.reportOn(source, FirErrors.EXPRESSION_EXPECTED_PACKAGE_FOUND, context)
            }
            else -> {}
        }
    }
}
