/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.explicitReceiverIsNotSuperReference
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.getSingleMatchedExpectForActualOrNull
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import ksp.org.jetbrains.kotlin.fir.expressions.impl.FirResolvedArgumentList
import ksp.org.jetbrains.kotlin.fir.references.toResolvedNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.unwrapFakeOverridesOrDelegated

object FirSuperCallWithDefaultsChecker : FirFunctionCallChecker(MppCheckerKind.Common) {
    override fun check(expression: FirFunctionCall, context: CheckerContext, reporter: DiagnosticReporter) {
        if (expression.explicitReceiverIsNotSuperReference()) return

        val functionSymbol = expression.calleeReference.toResolvedNamedFunctionSymbol() ?: return
        val relevantFunctionSymbol = functionSymbol.unwrapFakeOverridesOrDelegated()
            .let { it.getSingleMatchedExpectForActualOrNull() ?: it }
        if (!relevantFunctionSymbol.valueParameterSymbols.any { it.hasDefaultValue }) return
        val arguments = expression.argumentList as? FirResolvedArgumentList ?: return
        if (arguments.arguments.size < functionSymbol.valueParameterSymbols.size) {
            reporter.reportOn(
                expression.calleeReference.source,
                FirErrors.SUPER_CALL_WITH_DEFAULT_PARAMETERS,
                functionSymbol.name.asString(),
                context
            )
        }
    }
}
