/*
 * Copyright 2010-2022 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.findClosest
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirConstructor
import ksp.org.jetbrains.kotlin.fir.expressions.FirAnnotationCall
import ksp.org.jetbrains.kotlin.fir.expressions.FirArrayLiteral
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.resolvedType

object FirUnsupportedArrayLiteralChecker : FirArrayLiteralChecker(MppCheckerKind.Common) {
    override fun check(expression: FirArrayLiteral, context: CheckerContext, reporter: DiagnosticReporter) {
        if (!isInsideAnnotationCall(context) && !isInsideAnnotationConstructor(context)) {
            reporter.reportOn(
                expression.source,
                FirErrors.UNSUPPORTED,
                "Collection literals outside of annotations",
                context
            )
        }
    }

    private fun isInsideAnnotationCall(context: CheckerContext): Boolean = context.callsOrAssignments.asReversed().any {
        when (it) {
            is FirFunctionCall -> it.resolvedType.toRegularClassSymbol(context.session)?.classKind == ClassKind.ANNOTATION_CLASS
            is FirAnnotationCall -> true
            else -> false
        }
    }

    private fun isInsideAnnotationConstructor(context: CheckerContext): Boolean {
        return context.findClosest<FirConstructor>()?.returnTypeRef?.toRegularClassSymbol(context.session)?.classKind == ClassKind.ANNOTATION_CLASS
    }
}