/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.expression

import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.findClosest
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.explicitReceiverIsNotSuperReference
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirQualifiedAccessExpressionChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.isCompiledToJvmDefault
import ksp.org.jetbrains.kotlin.fir.declarations.*
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isInterface
import ksp.org.jetbrains.kotlin.fir.expressions.FirQualifiedAccessExpression
import ksp.org.jetbrains.kotlin.fir.java.jvmDefaultModeState
import ksp.org.jetbrains.kotlin.fir.references.toResolvedCallableSymbol
import ksp.org.jetbrains.kotlin.fir.resolve.providers.symbolProvider
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.name.SpecialNames.ANONYMOUS_FQ_NAME

object FirInterfaceDefaultMethodCallChecker : FirQualifiedAccessExpressionChecker(MppCheckerKind.Common) {
    override fun check(expression: FirQualifiedAccessExpression, context: CheckerContext, reporter: DiagnosticReporter) {
        if (context.languageVersionSettings.supportsFeature(LanguageFeature.AllowSuperCallToJavaInterface)) return

        val symbol = expression.calleeReference.toResolvedCallableSymbol()
        val classId = symbol?.callableId?.classId ?: return
        if (classId.isLocal) return

        if (expression.explicitReceiverIsNotSuperReference()) return

        val containingDeclaration = context.findClosest<FirRegularClass>() ?: return

        val session = context.session
        val typeSymbol = session.symbolProvider.getClassLikeSymbolByClassId(classId) as? FirRegularClassSymbol ?: return

        val jvmDefaultMode = session.jvmDefaultModeState
        if (typeSymbol.isInterface &&
            (typeSymbol.origin is FirDeclarationOrigin.Java || symbol.isCompiledToJvmDefault(session, jvmDefaultMode))
        ) {
            if (containingDeclaration.isInterface) {
                val containingMember = context.findContainingMember()?.symbol
                if (containingMember?.isCompiledToJvmDefault(session, jvmDefaultMode) == false) {
                    reporter.reportOn(expression.source, FirJvmErrors.INTERFACE_CANT_CALL_DEFAULT_METHOD_VIA_SUPER, context)
                    return
                }
            }
        }
    }

    private fun CheckerContext.findContainingMember(): FirCallableDeclaration? {
        return findClosest {
            (it is FirSimpleFunction && it.symbol.callableId.classId?.relativeClassName != ANONYMOUS_FQ_NAME) || it is FirProperty
        }
    }
}
