/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.jvm.checkers.expression

import ksp.org.jetbrains.kotlin.config.JvmTarget
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.FirElement
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirQualifiedAccessExpressionChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.isLhsOfAssignment
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import ksp.org.jetbrains.kotlin.fir.containingClassLookupTag
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isInline
import ksp.org.jetbrains.kotlin.fir.declarations.utils.sourceElement
import ksp.org.jetbrains.kotlin.fir.expressions.FirQualifiedAccessExpression
import ksp.org.jetbrains.kotlin.fir.expressions.toReference
import ksp.org.jetbrains.kotlin.fir.java.jvmTargetProvider
import ksp.org.jetbrains.kotlin.fir.references.toResolvedCallableSymbol
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.SymbolInternals
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirCallableSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirPropertySymbol
import ksp.org.jetbrains.kotlin.load.kotlin.FileBasedKotlinClass
import ksp.org.jetbrains.kotlin.load.kotlin.JvmPackagePartSource
import ksp.org.jetbrains.kotlin.load.kotlin.KotlinJvmBinarySourceElement

object FirJvmInlineTargetQualifiedAccessChecker : FirQualifiedAccessExpressionChecker(MppCheckerKind.Common) {
    override fun check(expression: FirQualifiedAccessExpression, context: CheckerContext, reporter: DiagnosticReporter) {
        val callableSymbol = expression.calleeReference.toResolvedCallableSymbol() ?: return
        if (callableSymbol.origin.fromSource) return

        val isInline = when (callableSymbol) {
            is FirFunctionSymbol<*> -> callableSymbol.isInline
            is FirPropertySymbol -> {
                val accessor = if (expression.isLhsOfAssignment(context)) callableSymbol.setterSymbol else callableSymbol.getterSymbol
                accessor != null && accessor.isInline
            }
            else -> false
        }

        if (isInline) {
            checkInlineTargetVersion(callableSymbol, context, reporter, expression)
        }
    }

    private fun checkInlineTargetVersion(
        callableSymbol: FirCallableSymbol<*>,
        context: CheckerContext,
        reporter: DiagnosticReporter,
        element: FirElement,
    ) {
        val currentJvmTarget = context.session.jvmTargetProvider?.jvmTarget ?: return

        val containingClass = callableSymbol.containingClassLookupTag()
        val binaryClass = if (containingClass != null) {
            val containingClassSymbol = containingClass.toRegularClassSymbol(context.session) ?: return

            @OptIn(SymbolInternals::class)
            val sourceElement = containingClassSymbol.fir.sourceElement as? KotlinJvmBinarySourceElement ?: return
            sourceElement.binaryClass
        } else {
            val containerSource = callableSymbol.containerSource as? JvmPackagePartSource ?: return
            containerSource.knownJvmBinaryClass
        }

        val inlinedVersion = (binaryClass as? FileBasedKotlinClass)?.classVersion ?: return
        val currentVersion = currentJvmTarget.majorVersion

        if (currentVersion < inlinedVersion) {
            reporter.reportOn(
                element.toReference(context.session)?.source ?: element.source,
                FirJvmErrors.INLINE_FROM_HIGHER_PLATFORM,
                JvmTarget.getDescription(inlinedVersion),
                JvmTarget.getDescription(currentVersion),
                context,
            )
        }
    }
}
