/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.native.checkers

import ksp.org.jetbrains.kotlin.KtRealSourceElementKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirBasicDeclarationChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.getModifier
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirDiagnosticRenderers
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirFunction
import ksp.org.jetbrains.kotlin.fir.declarations.FirMemberDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirTypeAlias
import ksp.org.jetbrains.kotlin.lexer.KtTokens

object FirNativeExternalDeclarationChecker : FirBasicDeclarationChecker(MppCheckerKind.Common) {
    override fun check(declaration: FirDeclaration, context: CheckerContext, reporter: DiagnosticReporter) {
        if (declaration !is FirMemberDeclaration) return
        // Functions are legal, typealias is checked by FirModifierChecker
        if (declaration is FirFunction || declaration is FirTypeAlias) return
        if (declaration.source?.kind !is KtRealSourceElementKind) return

        val modifier = declaration.getModifier(KtTokens.EXTERNAL_KEYWORD) ?: return

        reporter.reportOn(
            modifier.source,
            FirErrors.WRONG_MODIFIER_TARGET,
            modifier.token,
            FirDiagnosticRenderers.SYMBOL_KIND.render(declaration.symbol),
            context
        )
    }
}