/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.lazy

import ksp.org.jetbrains.kotlin.descriptors.DescriptorVisibility
import ksp.org.jetbrains.kotlin.descriptors.PropertyDescriptor
import ksp.org.jetbrains.kotlin.fir.backend.Fir2IrComponents
import ksp.org.jetbrains.kotlin.fir.backend.toIrType
import ksp.org.jetbrains.kotlin.fir.backend.utils.toIrConst
import ksp.org.jetbrains.kotlin.fir.declarations.FirField
import ksp.org.jetbrains.kotlin.fir.declarations.FirProperty
import ksp.org.jetbrains.kotlin.fir.declarations.FirRegularClass
import ksp.org.jetbrains.kotlin.fir.declarations.utils.*
import ksp.org.jetbrains.kotlin.fir.expressions.FirExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirLiteralExpression
import ksp.org.jetbrains.kotlin.fir.propertyIfBackingField
import ksp.org.jetbrains.kotlin.fir.unwrapFakeOverrides
import ksp.org.jetbrains.kotlin.fir.unwrapOr
import ksp.org.jetbrains.kotlin.ir.ObsoleteDescriptorBasedAPI
import ksp.org.jetbrains.kotlin.ir.declarations.IrDeclarationOrigin
import ksp.org.jetbrains.kotlin.ir.declarations.IrField
import ksp.org.jetbrains.kotlin.ir.declarations.MetadataSource
import ksp.org.jetbrains.kotlin.ir.declarations.createExpressionBody
import ksp.org.jetbrains.kotlin.ir.declarations.lazy.lazyVar
import ksp.org.jetbrains.kotlin.ir.expressions.IrConstructorCall
import ksp.org.jetbrains.kotlin.ir.expressions.IrExpressionBody
import ksp.org.jetbrains.kotlin.ir.symbols.IrFieldSymbol
import ksp.org.jetbrains.kotlin.ir.symbols.IrPropertySymbol
import ksp.org.jetbrains.kotlin.ir.types.IrType
import ksp.org.jetbrains.kotlin.name.Name

class Fir2IrLazyField(
    private val c: Fir2IrComponents,
    override val startOffset: Int,
    override val endOffset: Int,
    override var origin: IrDeclarationOrigin,
    override val fir: FirField,
    val containingClass: FirRegularClass?,
    override val symbol: IrFieldSymbol,
    correspondingPropertySymbol: IrPropertySymbol?
) : IrField(), AbstractFir2IrLazyDeclaration<FirField>, Fir2IrComponents by c {
    init {
        symbol.bind(this)
    }

    override var annotations: List<IrConstructorCall> by createLazyAnnotations()

    @ObsoleteDescriptorBasedAPI
    override val descriptor: PropertyDescriptor
        get() = symbol.descriptor

    override var isExternal: Boolean
        get() = fir.isExternal
        set(_) = mutationNotSupported()

    override var isFinal: Boolean
        get() = fir.isFinal
        set(_) = mutationNotSupported()

    override var isStatic: Boolean
        get() = fir.isStatic
        set(_) = mutationNotSupported()

    override var name: Name
        get() = fir.name
        set(_) = mutationNotSupported()

    override var visibility: DescriptorVisibility = c.visibilityConverter.convertToDescriptorVisibility(fir.visibility)
        set(_) = mutationNotSupported()

    override var type: IrType
        get() = fir.returnTypeRef.toIrType(c)
        set(_) = mutationNotSupported()

    override var initializer: IrExpressionBody? by lazyVar(lock) {
        val field = fir.unwrapFakeOverrides()
        val evaluatedInitializer = (field.propertyIfBackingField as? FirProperty)?.evaluatedInitializer?.unwrapOr<FirExpression> {}
        when (val initializer = evaluatedInitializer ?: field.initializer) {
            is FirLiteralExpression -> factory.createExpressionBody(initializer.toIrConst(type))
            else -> null
        }
    }

    override var correspondingPropertySymbol: IrPropertySymbol? = correspondingPropertySymbol
        set(_) = mutationNotSupported()

    override var metadata: MetadataSource?
        get() = null
        set(_) = error("We should never need to store metadata of external declarations.")
}
