/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.resolve.transformers

import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.FirFile
import ksp.org.jetbrains.kotlin.fir.declarations.FirResolvePhase
import ksp.org.jetbrains.kotlin.fir.resolve.ScopeSession
import ksp.org.jetbrains.kotlin.fir.symbols.lazyDeclarationResolver
import ksp.org.jetbrains.kotlin.fir.visitors.FirTransformer

@RequiresOptIn(message = "Should be used just only in resolve processor")
annotation class AdapterForResolveProcessor

sealed class FirResolveProcessor(val session: FirSession, val scopeSession: ScopeSession, val phase: FirResolvePhase?) {
    open fun beforePhase() {
        if (phase != null) {
            session.lazyDeclarationResolver.startResolvingPhase(phase)
        }
    }

    open fun afterPhase() {
        if (phase != null) {
            session.lazyDeclarationResolver.finishResolvingPhase(phase)
        }
    }
}

abstract class FirGlobalResolveProcessor(
    session: FirSession,
    scopeSession: ScopeSession,
    phase: FirResolvePhase,
) : FirResolveProcessor(session, scopeSession, phase) {
    abstract fun process(files: Collection<FirFile>)
}

abstract class FirTransformerBasedResolveProcessor(
    session: FirSession,
    scopeSession: ScopeSession,
    phase: FirResolvePhase?,
) : FirResolveProcessor(session, scopeSession, phase) {
    abstract val transformer: FirTransformer<Nothing?>

    open fun processFile(file: FirFile) {
        file.transform<FirFile, Nothing?>(transformer, null)
    }
}
