/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.backend.js.lower

import ksp.org.jetbrains.kotlin.backend.common.DeclarationTransformer
import ksp.org.jetbrains.kotlin.descriptors.DescriptorVisibilities
import ksp.org.jetbrains.kotlin.descriptors.Modality
import ksp.org.jetbrains.kotlin.ir.backend.js.JsIrBackendContext
import ksp.org.jetbrains.kotlin.ir.backend.js.defaultConstructorForReflection
import ksp.org.jetbrains.kotlin.ir.declarations.*
import ksp.org.jetbrains.kotlin.ir.util.constructors
import ksp.org.jetbrains.kotlin.ir.util.isClass

/**
 * Collects classes default constructors to add it to metadata on code generating phase.
 */
class CollectClassDefaultConstructorsLowering(private val context: JsIrBackendContext) : DeclarationTransformer {
    override fun transformFlat(declaration: IrDeclaration): List<IrDeclaration>? {
        if (declaration is IrClass && declaration.couldContainDefaultConstructor()) {
            declaration.defaultConstructorForReflection = declaration.defaultConstructor ?: return null
        }

        return null
    }

    private fun IrClass.couldContainDefaultConstructor(): Boolean {
        return isClass && !isValue && !isExpect && modality != Modality.ABSTRACT && modality != Modality.SEALED
    }

    private val IrClass.defaultConstructor: IrConstructor?
        get() = constructors.singleOrNull { it.visibility == DescriptorVisibilities.PUBLIC && it.isDefaultConstructor() }

    private fun IrFunction.isDefaultConstructor(): Boolean {
        return valueParameters.isEmpty() || valueParameters.all { it.defaultValue != null }
    }
}
