/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.backend.js.lower

import ksp.org.jetbrains.kotlin.backend.common.BodyLoweringPass
import ksp.org.jetbrains.kotlin.ir.backend.js.JsCommonBackendContext
import ksp.org.jetbrains.kotlin.ir.backend.js.JsStatementOrigins
import ksp.org.jetbrains.kotlin.ir.declarations.IrConstructor
import ksp.org.jetbrains.kotlin.ir.declarations.IrDeclaration
import ksp.org.jetbrains.kotlin.ir.expressions.IrBody
import ksp.org.jetbrains.kotlin.ir.expressions.IrStatementContainer
import ksp.org.jetbrains.kotlin.ir.expressions.impl.IrCallImpl
import ksp.org.jetbrains.kotlin.ir.util.*

/**
 * Invokes companion object's initializers from companion object in object constructor.
 */
class InvokeStaticInitializersLowering(val context: JsCommonBackendContext) : BodyLoweringPass {
    override fun lower(irBody: IrBody, container: IrDeclaration) {
        if (container !is IrConstructor) return
        if (container.parentClassOrNull?.isEnumClass == true) return

        val irClass = container.constructedClass
        if (irClass.isEffectivelyExternal()) {
            return
        }

        val companionObject = irClass.companionObject() ?: return

        val instance = context.mapping.objectToGetInstanceFunction[companionObject] ?: return

        val getInstanceCall = IrCallImpl(
            irClass.startOffset,
            irClass.endOffset,
            context.irBuiltIns.unitType,
            instance.symbol,
            typeArgumentsCount = 0,
            origin = JsStatementOrigins.SYNTHESIZED_STATEMENT
        )

        (irBody as IrStatementContainer).statements.add(0, getInstanceCall)
    }
}
