/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.backend.js.utils

import ksp.org.jetbrains.kotlin.ir.IrElement
import ksp.org.jetbrains.kotlin.ir.backend.js.JsCommonBackendContext
import ksp.org.jetbrains.kotlin.ir.backend.js.JsIrBackendContext
import ksp.org.jetbrains.kotlin.ir.backend.js.JsLoweredDeclarationOrigin
import ksp.org.jetbrains.kotlin.ir.backend.js.export.isExported
import ksp.org.jetbrains.kotlin.ir.backend.js.ir.JsIrBuilder
import ksp.org.jetbrains.kotlin.ir.declarations.*
import ksp.org.jetbrains.kotlin.ir.expressions.*
import ksp.org.jetbrains.kotlin.ir.types.getClass
import ksp.org.jetbrains.kotlin.ir.types.isNullableAny
import ksp.org.jetbrains.kotlin.ir.util.invokeFun
import ksp.org.jetbrains.kotlin.ir.util.isEffectivelyExternal
import ksp.org.jetbrains.kotlin.ir.util.isMethodOfAny
import ksp.org.jetbrains.kotlin.ir.util.isTopLevel
import ksp.org.jetbrains.kotlin.ir.util.isTopLevelDeclaration
import ksp.org.jetbrains.kotlin.js.config.JSConfigurationKeys
import ksp.org.jetbrains.kotlin.util.OperatorNameConventions

fun TODO(element: IrElement): Nothing = TODO(element::class.java.simpleName + " is not supported yet here")

fun IrFunction.hasStableJsName(context: JsIrBackendContext): Boolean {
    if (
        origin == JsLoweredDeclarationOrigin.BRIDGE_WITH_STABLE_NAME ||
        (this as? IrSimpleFunction)?.isMethodOfAny() == true // Handle names for special functions
    ) {
        return true
    }

    if (
        origin == JsLoweredDeclarationOrigin.JS_SHADOWED_EXPORT ||
        origin == JsLoweredDeclarationOrigin.BRIDGE_WITHOUT_STABLE_NAME ||
        origin == JsLoweredDeclarationOrigin.BRIDGE_PROPERTY_ACCESSOR
    ) {
        return false
    }

    val namedOrMissingGetter = when (this) {
        is IrSimpleFunction -> {
            val owner = correspondingPropertySymbol?.owner
            if (owner == null) {
                true
            } else {
                owner.getter?.getJsName() != null
            }
        }
        is IrConstructor -> true
    }

    return (isEffectivelyExternal() || getJsName() != null || isExported(context)) && namedOrMissingGetter
}

fun IrFunction.isEqualsInheritedFromAny(): Boolean =
    name == OperatorNameConventions.EQUALS &&
            dispatchReceiverParameter != null &&
            extensionReceiverParameter == null &&
            valueParameters.size == 1 &&
            valueParameters[0].type.isNullableAny()

fun IrDeclaration.hasStaticDispatch() = when (this) {
    is IrSimpleFunction -> dispatchReceiverParameter == null
    is IrProperty -> isTopLevelDeclaration
    is IrField -> isStatic
    else -> true
}

val IrValueDeclaration.isDispatchReceiver: Boolean
    get() {
        val parent = this.parent
        if (parent is IrClass)
            return true
        if (parent is IrFunction && parent.dispatchReceiverParameter == this)
            return true
        return false
    }

fun IrBody.prependFunctionCall(
    call: IrCall
) {
    when (this) {
        is IrExpressionBody -> {
            expression = JsIrBuilder.buildComposite(
                type = expression.type,
                statements = listOf(
                    call,
                    expression
                )
            )
        }
        is IrBlockBody -> {
            statements.add(
                0,
                call
            )
        }
        is IrSyntheticBody -> Unit
    }
}

fun JsCommonBackendContext.findUnitGetInstanceFunction(): IrSimpleFunction =
    mapping.objectToGetInstanceFunction[irBuiltIns.unitClass.owner]!!

fun JsCommonBackendContext.findUnitInstanceField(): IrField =
    mapping.objectToInstanceField[irBuiltIns.unitClass.owner]!!

val JsCommonBackendContext.compileSuspendAsJsGenerator: Boolean
    get() = this is JsIrBackendContext && configuration[JSConfigurationKeys.COMPILE_SUSPEND_AS_JS_GENERATOR] == true

fun IrDeclaration.isImportedFromModuleOnly(): Boolean {
    return isTopLevel && isEffectivelyExternal() && (getJsModule() != null && !isJsNonModule() || (parent as? IrAnnotationContainer)?.getJsModule() != null)
}

fun invokeFunForLambda(call: IrCall) =
    call.extensionReceiver!!
        .type
        .getClass()!!
        .invokeFun!!