/*
 * Copyright 2010-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ksp.org.jetbrains.kotlin.js.translate.intrinsic.functions.factories

import ksp.org.jetbrains.kotlin.builtins.StandardNames
import ksp.org.jetbrains.kotlin.descriptors.ConstructorDescriptor
import ksp.org.jetbrains.kotlin.descriptors.FunctionDescriptor
import ksp.org.jetbrains.kotlin.js.backend.ast.JsExpression
import ksp.org.jetbrains.kotlin.js.backend.ast.JsInvocation
import ksp.org.jetbrains.kotlin.js.translate.callTranslator.CallInfo
import ksp.org.jetbrains.kotlin.js.translate.context.Namer
import ksp.org.jetbrains.kotlin.js.translate.context.TranslationContext
import ksp.org.jetbrains.kotlin.js.translate.intrinsic.functions.basic.FunctionIntrinsic
import ksp.org.jetbrains.kotlin.js.translate.utils.JsAstUtils
import ksp.org.jetbrains.kotlin.resolve.descriptorUtil.fqNameSafe
import ksp.org.jetbrains.kotlin.types.typeUtil.isNotNullThrowable

object ThrowableConstructorIntrinsicFactory : FunctionIntrinsicFactory {
    override fun getIntrinsic(descriptor: FunctionDescriptor, context: TranslationContext): FunctionIntrinsic? {
        if (descriptor !is ConstructorDescriptor) return null
        if (!descriptor.constructedClass.defaultType.isNotNullThrowable()) return null

        return Intrinsic
    }

    object Intrinsic : FunctionIntrinsic() {
        override fun apply(callInfo: CallInfo, arguments: List<JsExpression>, context: TranslationContext): JsExpression {
            val constructor = callInfo.resolvedCall.resultingDescriptor
            val argumentsToPass = arguments.toMutableList()
            val hasCauseParameter = constructor.valueParameters.any {
                it.type.constructor.declarationDescriptor?.fqNameSafe == StandardNames.FqNames.throwable
            }

            if (constructor.valueParameters.size == 1 && hasCauseParameter) {
                argumentsToPass.add(0, Namer.getUndefinedExpression())
            }

            return JsInvocation(JsAstUtils.pureFqn("newThrowable", Namer.kotlinObject()), argumentsToPass)
        }
    }
}
