/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.psi

import ksp.com.intellij.lang.ASTNode
import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.annotations.NonNls
import ksp.org.jetbrains.kotlin.lexer.KtTokens
import ksp.org.jetbrains.kotlin.name.SpecialNames
import ksp.org.jetbrains.kotlin.psi.stubs.KotlinObjectStub
import ksp.org.jetbrains.kotlin.psi.stubs.elements.KtStubElementTypes

class KtObjectDeclaration : KtClassOrObject {
    constructor(node: ASTNode) : super(node)
    constructor(stub: KotlinObjectStub) : super(stub, KtStubElementTypes.OBJECT_DECLARATION)

    private val _stub: KotlinObjectStub?
        get() = greenStub as? KotlinObjectStub

    override fun getName(): String? {
        super.getName()?.let { return it }

        if (isCompanion() && !isTopLevel()) {
            //NOTE: a hack in PSI that simplifies writing frontend code
            return SpecialNames.DEFAULT_NAME_FOR_COMPANION_OBJECT.toString()
        }

        return null
    }

    override fun setName(@NonNls name: String): PsiElement {
        return if (nameIdentifier == null) {
            val psiFactory = KtPsiFactory(project)
            val result = addAfter(psiFactory.createIdentifier(name), getObjectKeyword()!!)
            addAfter(psiFactory.createWhiteSpace(), getObjectKeyword()!!)

            result
        } else {
            super.setName(name)
        }
    }

    fun isCompanion(): Boolean = _stub?.isCompanion() ?: hasModifier(KtTokens.COMPANION_KEYWORD)

    override fun getTextOffset(): Int = nameIdentifier?.textRange?.startOffset
        ?: getObjectKeyword()!!.textRange.startOffset

    override fun <R, D> accept(visitor: KtVisitor<R, D>, data: D): R {
        return visitor.visitObjectDeclaration(this, data)
    }

    fun isObjectLiteral(): Boolean = _stub?.isObjectLiteral() ?: (parent is KtObjectLiteralExpression)

    fun getObjectKeyword(): PsiElement? = findChildByType(KtTokens.OBJECT_KEYWORD)

    override fun getIdentifyingElement(): PsiElement? = getObjectKeyword()

    override fun getCompanionObjects(): List<KtObjectDeclaration> = emptyList()
}
