/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.psi2ir.generators

import ksp.org.jetbrains.kotlin.backend.common.SamTypeApproximator
import ksp.org.jetbrains.kotlin.builtins.ReflectionTypes
import ksp.org.jetbrains.kotlin.config.LanguageVersionSettings
import ksp.org.jetbrains.kotlin.descriptors.CallableDescriptor
import ksp.org.jetbrains.kotlin.descriptors.ModuleDescriptor
import ksp.org.jetbrains.kotlin.descriptors.NotFoundClasses
import ksp.org.jetbrains.kotlin.ir.IrBuiltIns
import ksp.org.jetbrains.kotlin.ir.builders.IrGeneratorContext
import ksp.org.jetbrains.kotlin.ir.expressions.IrDeclarationReference
import ksp.org.jetbrains.kotlin.ir.util.SymbolTable
import ksp.org.jetbrains.kotlin.ir.util.TypeTranslator
import ksp.org.jetbrains.kotlin.psi.KtFile
import ksp.org.jetbrains.kotlin.psi2ir.Psi2IrConfiguration
import ksp.org.jetbrains.kotlin.psi2ir.generators.fragments.FragmentContext
import ksp.org.jetbrains.kotlin.resolve.BindingContext
import ksp.org.jetbrains.kotlin.storage.LockBasedStorageManager

class GeneratorContext private constructor(
    val configuration: Psi2IrConfiguration,
    val moduleDescriptor: ModuleDescriptor,
    val bindingContext: BindingContext,
    val languageVersionSettings: LanguageVersionSettings,
    val symbolTable: SymbolTable,
    val extensions: GeneratorExtensions,
    val typeTranslator: TypeTranslator,
    override val irBuiltIns: IrBuiltIns,
    internal val callToSubstitutedDescriptorMap: MutableMap<IrDeclarationReference, CallableDescriptor>,
    internal var fragmentContext: FragmentContext?,
) : IrGeneratorContext {

    constructor(
        configuration: Psi2IrConfiguration,
        moduleDescriptor: ModuleDescriptor,
        bindingContext: BindingContext,
        languageVersionSettings: LanguageVersionSettings,
        symbolTable: SymbolTable,
        extensions: GeneratorExtensions,
        typeTranslator: TypeTranslator,
        irBuiltIns: IrBuiltIns,
        fragmentContext: FragmentContext? = null,
    ) : this(
        configuration,
        moduleDescriptor,
        bindingContext,
        languageVersionSettings,
        symbolTable,
        extensions,
        typeTranslator,
        irBuiltIns,
        mutableMapOf(),
        fragmentContext,
    )

    val constantValueGenerator = typeTranslator.constantValueGenerator

    fun IrDeclarationReference.commitSubstituted(descriptor: CallableDescriptor) {
        callToSubstitutedDescriptorMap[this] = descriptor
    }

    // TODO: inject a correct StorageManager instance, or store NotFoundClasses inside ModuleDescriptor
    val reflectionTypes = ReflectionTypes(moduleDescriptor, NotFoundClasses(LockBasedStorageManager.NO_LOCKS, moduleDescriptor))

    internal val additionalDescriptorStorage: DescriptorStorageForContextReceivers = DescriptorStorageForContextReceivers()

    val samTypeApproximator = SamTypeApproximator(moduleDescriptor.builtIns, languageVersionSettings)

    fun createFileScopeContext(ktFile: KtFile): GeneratorContext {
        return GeneratorContext(
            configuration,
            moduleDescriptor,
            bindingContext,
            languageVersionSettings,
            symbolTable,
            extensions,
            TypeTranslatorImpl(
                symbolTable, languageVersionSettings, moduleDescriptor, extensions = extensions, ktFile = ktFile,
                allowErrorTypeInAnnotations = configuration.skipBodies,
            ),
            irBuiltIns,
            callToSubstitutedDescriptorMap,
            fragmentContext,
        )
    }
}
