/*
 * Copyright 2010-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ksp.org.jetbrains.kotlin.resolve.calls.model;

import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;
import ksp.org.jetbrains.kotlin.descriptors.CallableDescriptor;
import ksp.org.jetbrains.kotlin.descriptors.TypeParameterDescriptor;
import ksp.org.jetbrains.kotlin.descriptors.ValueParameterDescriptor;
import ksp.org.jetbrains.kotlin.psi.Call;
import ksp.org.jetbrains.kotlin.psi.ValueArgument;
import ksp.org.jetbrains.kotlin.resolve.calls.results.ResolutionStatus;
import ksp.org.jetbrains.kotlin.resolve.calls.tasks.ExplicitReceiverKind;
import ksp.org.jetbrains.kotlin.resolve.scopes.receivers.ReceiverValue;
import ksp.org.jetbrains.kotlin.types.KotlinType;

import java.util.List;
import java.util.Map;

public abstract class DelegatingResolvedCall<D extends CallableDescriptor> implements ResolvedCall<D> {
    private final ResolvedCall<? extends D> resolvedCall;

    public DelegatingResolvedCall(@NotNull ResolvedCall<? extends D> resolvedCall) {
        this.resolvedCall = resolvedCall;
    }

    @NotNull
    @Override
    public ResolutionStatus getStatus() {
        return resolvedCall.getStatus();
    }

    @NotNull
    @Override
    public Call getCall() {
        return resolvedCall.getCall();
    }

    @NotNull
    @Override
    public D getCandidateDescriptor() {
        return resolvedCall.getCandidateDescriptor();
    }

    @NotNull
    @Override
    public D getResultingDescriptor() {
        return resolvedCall.getResultingDescriptor();
    }

    @Nullable
    @Override
    public ReceiverValue getExtensionReceiver() {
        return resolvedCall.getExtensionReceiver();
    }

    @Nullable
    @Override
    public ReceiverValue getDispatchReceiver() {
        return resolvedCall.getDispatchReceiver();
    }

    @NotNull
    @Override
    public List<ReceiverValue> getContextReceivers() {
        return resolvedCall.getContextReceivers();
    }

    @NotNull
    @Override
    public ExplicitReceiverKind getExplicitReceiverKind() {
        return resolvedCall.getExplicitReceiverKind();
    }

    @NotNull
    @Override
    public Map<ValueParameterDescriptor, ResolvedValueArgument> getValueArguments() {
        return resolvedCall.getValueArguments();
    }

    @NotNull
    @Override
    public ArgumentMapping getArgumentMapping(@NotNull ValueArgument valueArgument) {
        return resolvedCall.getArgumentMapping(valueArgument);
    }

    @Nullable
    @Override
    public List<ResolvedValueArgument> getValueArgumentsByIndex() {
        return resolvedCall.getValueArgumentsByIndex();
    }

    @NotNull
    @Override
    public Map<TypeParameterDescriptor, KotlinType> getTypeArguments() {
        return resolvedCall.getTypeArguments();
    }

    @NotNull
    @Override
    public DataFlowInfoForArguments getDataFlowInfoForArguments() {
        return resolvedCall.getDataFlowInfoForArguments();
    }

    @Nullable
    @Override
    public KotlinType getSmartCastDispatchReceiverType() {
        return resolvedCall.getSmartCastDispatchReceiverType();
    }
}
