/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.resolve.jvm.checkers

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.resolve.calls.checkers.CallChecker
import ksp.org.jetbrains.kotlin.resolve.calls.checkers.CallCheckerContext
import ksp.org.jetbrains.kotlin.resolve.calls.model.ResolvedCall
import ksp.org.jetbrains.kotlin.resolve.calls.model.VarargValueArgument
import ksp.org.jetbrains.kotlin.resolve.jvm.diagnostics.ErrorsJvm

object PolymorphicSignatureCallChecker : CallChecker {
    @JvmField
    val polymorphicSignatureFqName = FqName("java.lang.invoke.MethodHandle.PolymorphicSignature")

    override fun check(resolvedCall: ResolvedCall<*>, reportOn: PsiElement, context: CallCheckerContext) {
        if (!context.languageVersionSettings.supportsFeature(LanguageFeature.PolymorphicSignature)) return
        if (!resolvedCall.resultingDescriptor.annotations.hasAnnotation(polymorphicSignatureFqName)) return

        for (valueArgument in resolvedCall.valueArgumentsByIndex ?: return) {
            if (valueArgument !is VarargValueArgument) continue
            for (argument in valueArgument.arguments) {
                val spread = argument.getSpreadElement() ?: continue
                context.trace.report(ErrorsJvm.SPREAD_ON_SIGNATURE_POLYMORPHIC_CALL.on(context.languageVersionSettings, spread))
            }
        }
    }
}
