/*
 * Decompiled with CFR 0.152.
 */
package com.google.api.server.spi.auth;

import com.google.api.client.http.GenericUrl;
import com.google.api.client.http.HttpIOExceptionHandler;
import com.google.api.client.http.HttpRequest;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.http.HttpUnsuccessfulResponseHandler;
import com.google.api.client.util.Key;
import com.google.api.server.spi.Client;
import com.google.api.server.spi.Strings;
import com.google.api.server.spi.response.ServiceUnavailableException;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.flogger.FluentLogger;
import java.io.IOException;
import java.util.List;
import java.util.regex.Pattern;
import javax.servlet.http.HttpServletRequest;

public class GoogleAuth {
    private static final FluentLogger logger = FluentLogger.forEnclosingClass();
    private static final String BASE64_REGEX = "[a-zA-Z0-9+/=_-]{6,}+";
    private static final Pattern JWT_PATTERN = Pattern.compile(String.format("%s\\.%s\\.%s", "[a-zA-Z0-9+/=_-]{6,}+", "[a-zA-Z0-9+/=_-]{6,}+", "[a-zA-Z0-9+/=_-]{6,}+"));
    private static final String TOKEN_INFO_ENDPOINT = "https://www.googleapis.com/oauth2/v2/tokeninfo";
    @VisibleForTesting
    static final String AUTHORIZATION_HEADER = "Authorization";
    @VisibleForTesting
    static final String[] ALLOWED_AUTH_SCHEMES = new String[]{"Bearer", "OAuth"};
    @VisibleForTesting
    static final String[] BEARER_TOKEN_PARAMETER_NAMES = new String[]{"access_token", "bearer_token"};
    @VisibleForTesting
    static final String[] OAUTH2_TOKEN_PREFIXES = new String[]{"ya29.", "1/"};
    @VisibleForTesting
    static final List<String> SKIP_CLIENT_ID_CHECK_LIST = ImmutableList.of((Object)"*");

    public static String getAuthToken(HttpServletRequest request) {
        if (request.getAttribute("endpoints:Auth-Token") == null) {
            String token = GoogleAuth.getAuthTokenFromHeader(request.getHeader(AUTHORIZATION_HEADER));
            if (token == null) {
                token = GoogleAuth.getAuthTokenFromQueryParameters(request);
            }
            request.setAttribute("endpoints:Auth-Token", (Object)token);
        }
        return (String)request.getAttribute("endpoints:Auth-Token");
    }

    private static String getAuthTokenFromQueryParameters(HttpServletRequest request) {
        for (String parameterName : BEARER_TOKEN_PARAMETER_NAMES) {
            String token = request.getParameter(parameterName);
            if (token == null) continue;
            return token;
        }
        return null;
    }

    private static String getAuthTokenFromHeader(String authHeader) {
        String authScheme = GoogleAuth.matchAuthScheme(authHeader);
        if (authScheme == null || authScheme.length() >= authHeader.length()) {
            return null;
        }
        return authHeader.substring(authScheme.length() + 1);
    }

    private static String matchAuthScheme(String authHeader) {
        if (authHeader == null) {
            return null;
        }
        for (String authScheme : ALLOWED_AUTH_SCHEMES) {
            if (!authHeader.startsWith(authScheme)) continue;
            return authScheme;
        }
        return null;
    }

    public static boolean isJwt(String token) {
        if (token == null) {
            return false;
        }
        return JWT_PATTERN.matcher(token).matches();
    }

    public static boolean isOAuth2Token(String token) {
        if (token == null) {
            return false;
        }
        String strippedToken = token.trim().replaceFirst("^['\"]", "");
        for (String prefix : OAUTH2_TOKEN_PREFIXES) {
            if (!strippedToken.startsWith(prefix)) continue;
            return true;
        }
        return false;
    }

    static boolean checkClientId(String clientId, List<String> allowedClientIds, boolean allowSkipClientIdCheck) {
        if (Strings.isWhitelisted(clientId, allowedClientIds)) {
            return true;
        }
        return allowSkipClientIdCheck && !Strings.isEmptyOrNull(allowedClientIds) && allowedClientIds.equals(SKIP_CLIENT_ID_CHECK_LIST);
    }

    static boolean checkAudience(String audience, List<String> allowedAudiences, String clientId) {
        if (Strings.isWhitelisted(audience, allowedAudiences)) {
            return true;
        }
        return !Strings.isEmptyOrWhitespace(audience) && audience.equals(clientId);
    }

    public static TokenInfo getTokenInfoRemote(String token) throws ServiceUnavailableException {
        try {
            String tokenParam;
            if (GoogleAuth.isOAuth2Token(token)) {
                tokenParam = "?access_token=";
            } else if (GoogleAuth.isJwt(token)) {
                tokenParam = "?id_token=";
            } else {
                return null;
            }
            HttpRequest request = Client.getInstance().getJsonHttpRequestFactory().buildGetRequest(new GenericUrl(TOKEN_INFO_ENDPOINT + tokenParam + token));
            GoogleAuth.configureErrorHandling(request);
            return GoogleAuth.parseTokenInfo(request);
        }
        catch (IOException e) {
            throw new ServiceUnavailableException("Failed to perform access token validation", (Throwable)e);
        }
    }

    @VisibleForTesting
    static TokenInfo parseTokenInfo(HttpRequest request) throws IOException, ServiceUnavailableException {
        HttpResponse response = request.execute();
        int statusCode = response.getStatusCode();
        TokenInfo info = (TokenInfo)response.parseAs(TokenInfo.class);
        if (statusCode != 200) {
            String errorDescription = "Unknown error";
            if (info != null && info.errorDescription != null) {
                errorDescription = info.errorDescription;
            }
            errorDescription = errorDescription + " (" + statusCode + ")";
            if (statusCode >= 500) {
                ((FluentLogger.Api)logger.atSevere()).log("Error validating access token: %s", (Object)errorDescription);
                throw new ServiceUnavailableException("Failed to validate access token");
            }
            ((FluentLogger.Api)logger.atInfo()).log("Invalid access token: %s", (Object)errorDescription);
            return null;
        }
        if (info == null || Strings.isEmptyOrWhitespace(info.email)) {
            ((FluentLogger.Api)logger.atWarning()).log("Access token does not contain email scope");
            return null;
        }
        return info;
    }

    @VisibleForTesting
    static void configureErrorHandling(HttpRequest request) {
        request.setNumberOfRetries(1).setThrowExceptionOnExecuteError(false).setIOExceptionHandler(new HttpIOExceptionHandler(){

            public boolean handleIOException(HttpRequest request, boolean supportsRetry) {
                return true;
            }
        }).setUnsuccessfulResponseHandler(new HttpUnsuccessfulResponseHandler(){

            public boolean handleResponse(HttpRequest request, HttpResponse response, boolean supportsRetry) {
                return response.getStatusCode() >= 500;
            }
        });
    }

    public static class TokenInfo {
        @Key(value="email")
        public String email;
        @Key(value="issued_to")
        public String clientId;
        @Key(value="scope")
        public String scopes;
        @Key(value="user_id")
        public String userId;
        @Key(value="audience")
        public String audience;
        @Key(value="expires_in")
        public Integer expiresIn;
        @Key(value="verified_email")
        public Boolean verifiedEmail;
        @Key(value="error_description")
        public String errorDescription;
    }
}

